#!/usr/bin/env python3
from syslog import syslog, openlog, closelog, LOG_ERR, LOG_INFO
import sys

version = '.'.join([str(x) for x in sys.version_info[:2]])
sys.path.append('/usr/local/lib/python%s/site-packages' % version)
sys.path.append('/usr/local/lib64/python%s/site-packages' % version)

# Set python cache for rrdtool
import os
os.environ['PYTHON_EGG_CACHE'] = '/usr/local/nagiosna/tmp'

import subprocess
import tempfile
import config
import rrdtool
import re
import initialize_source
import traceback
# import netflow_checks
import logging
import logging.handlers
import datetime

logger = logging.getLogger('backend')
hdlr = logging.handlers.RotatingFileHandler('/usr/local/nagiosna/var/backend.log', 'a', 1048576, 10)

fmt = logging.Formatter('%(asctime)s %(levelname)s : %(message)s', '%Y-%m-%d %H:%M:%S')
hdlr.setFormatter(fmt)

logger.addHandler(hdlr)
logger.setLevel(logging.DEBUG)

class Dummy(object): 
    pass

nfdump_cmd = ['/usr/local/bin/nfdump']
nfexpire_cmd = ['/usr/local/bin/nfexpire']
nfreplay_cmd = ['/usr/local/bin/nfreplay']

def format_summary_to_csv(nfdump_summary):
    # Extract the first line which contains all the metrics we need
    summary_line = nfdump_summary.split('\n')[0]
    
    values = {
        'flows': '0',
        'bytes': '0',
        'packets': '0',
        'avg_bps': '0',
        'avg_pps': '0',
        'avg_bpp': '0'
    }

    for item in summary_line.split(','):
        item = item.strip()
        if 'flows:' in item:
            values['flows'] = item.split('flows:')[1].strip()
        elif 'bytes:' in item:
            values['bytes'] = item.split('bytes:')[1].strip()
        elif 'packets:' in item:
            values['packets'] = item.split('packets:')[1].strip()
        elif 'avg bps:' in item:
            values['avg_bps'] = item.split('avg bps:')[1].strip()
        elif 'avg pps:' in item:
            values['avg_pps'] = item.split('avg pps:')[1].strip()
        elif 'avg bpp:' in item:
            values['avg_bpp'] = item.split('avg bpp:')[1].strip()
    
    # Create CSV header and data rows
    header = "\n\nSummary\nflows,bytes,packets,avg_bps,avg_pps,avg_bpp"
    data_row = f"{values['flows']},{values['bytes']},{values['packets']},{values['avg_bps']},{values['avg_pps']},{values['avg_bpp']}"
    
    return f"{header}\n{data_row}"

def update_bandwidth(directory, ABS_FLOWFILE, path):
    global nfdump_cmd
    global logger
    bandwidth_rrd = os.path.abspath(os.path.join(directory, *path))

    csv_cmd = nfdump_cmd + ['-r', ABS_FLOWFILE, '-s', 'dstport', '-n', '1', '-N', '-o', 'csv']
    csv_update = subprocess.Popen(csv_cmd, stdout=subprocess.PIPE)
    csv_update.wait()
    csv_output = csv_update.stdout.readlines()

    summary_cmd = nfdump_cmd + ['-r', ABS_FLOWFILE, '-s', 'dstport', '-n', '1', '-N', '-o', 'null']
    summary_update = subprocess.Popen(summary_cmd, stdout=subprocess.PIPE)
    summary_update.wait()
    #format_summary_to_csv requires a string so we need to convert to string
    summary_string = ''.join(line.decode('utf-8') for line in summary_update.stdout.readlines())
    formatted_summary = format_summary_to_csv(summary_string)
    formatted_summary_lines = formatted_summary.split('\n')

    # Combine outputs, readlines returns in bytes so we need to convert formatted summary back
    stdout = csv_output + [line.encode('utf-8') + b'\n' for line in formatted_summary_lines if line]

    summary = ['N'] + stdout[-1].decode('utf-8').rstrip().split(',')
    rrdstamp = str(':'.join(summary))

    try:
        rrdtool.update(bandwidth_rrd, rrdstamp)
    except Exception as e:
        logging.warning('Error updating the RRD: bandwidth rrd was not updated: ' + str(e))
        
# todo Uncomment and refactor this when running checks if we need.
# def run_checks(ABS_FLOWFILE, sid):
#     global logger
#     logger.debug('Running checks...')
#     options = Dummy()
#     options.filename = ABS_FLOWFILE
#     options.sourceid = sid
#     try:
#         netflow_checks.main(options)
#         logger.info('Ran checks successfully')
#     except Exception as e:
#         logger.exception(e)
    
    
def test_sanity(ABS_FLOWFILE):
    assert os.path.isfile(ABS_FLOWFILE)

# todo Uncomment and recator this when views are added
# def update_views(directory, flowfile, sid, ABS_FLOWFILE):
#     global logger
#     global nfdump_cmd
#     global nfexpire_cmd
#     ABS_VIEWPATH = os.path.join(directory, '..', 'views')
#     QUERY_SQL = '''
# SELECT 
#     nagiosna_Views.name, 
#     nagiosna_Views.limiter,
#     nagiosna_Views.lifetime
# FROM 
#     nagiosna_Views
# JOIN 
#     nagiosna_SourcesViewsLinker 
# ON 
#     nagiosna_SourcesViewsLinker.vid = nagiosna_Views.vid
# WHERE
#     nagiosna_SourcesViewsLinker.sid = %s;''' % sid 
#     views = config.query_database(QUERY_SQL)
#     for view in views:
#         name, limiter, lifetime = view
#         name = config.safe_filename(name)
#         target_dir = os.path.join(ABS_VIEWPATH, name)
#         if not os.path.isdir(target_dir):
#             logger.warning('Target dir %s, not present. Creating it...' % target_dir)
#             os.makedirs(target_dir)
#         target = os.path.join(target_dir, flowfile)
#         nfdc = nfdump_cmd + ['-r', ABS_FLOWFILE, '-w', target, limiter]
#         nfex = nfexpire_cmd + ['-t', lifetime, '-e', os.path.join(ABS_VIEWPATH, name)]
#         nfdump = subprocess.Popen(nfdc)
#         nfexpire = subprocess.Popen(nfex)
#         nfdump.wait()
#         nfexpire.wait()
        
#         if nfdump.returncode != 0:
#             logger.error("nfdump command: %s, did not finish properly" % ' '.join(nfcd))
#         if nfexpire.returncode != 0:
#             logger.error("nfexpire command: %s, did not finish properly" % ' '.join(nfex))
        
#         path = view_rrd_path(ABS_VIEWPATH, name)
        
#         update_bandwidth(directory, target, path)

# returns a list that can be os.path.join'd
# Not the best way but we're just refactoring for what's already here.
def view_rrd_path(abs_path, name):
    file_name = '%s-bandwidth.rrd' % name
    if not os.path.isfile(os.path.join(abs_path, file_name)):
        logger.warning('Bandwidth RRD does not seem to exist, creating it...')
        initialize_source.create_bandwidth_db(abs_path, file_name)
    return [abs_path, file_name]

def main():
    global logger
    #~ We __must__ get the arguments passed from the nfcapd/sfcapd process
    #~ so don't even catch the exception if it gets thrown.
    directory = sys.argv[1]
    flowfile  = sys.argv[2]
    sid = sys.argv[3]
    
    logger.info('Parsing data for the source id: %s' % str(sid))
    logger.debug('Arguments: %s, %s, %i' % (directory, flowfile, int(sid)))
    
    ABS_FLOWFILE = os.path.join(directory, flowfile)
    
    test_sanity(ABS_FLOWFILE)
    mainrrd = ['..', 'bandwidth.rrd']
    try:
        update_bandwidth(directory, ABS_FLOWFILE, mainrrd)
    except Exception as e:
        logger.exception(e)
        raise e
    # update_views(directory, flowfile, sid, ABS_FLOWFILE)
    # run_checks(ABS_FLOWFILE, sid)

if __name__ == "__main__":
    openlog('backend')
    try:
        main()
        logger.info("Successfully reaped nfcapd file.")
    except Exception as e:
        etype, evalue, tback = sys.exc_info()
        logger.error("Error reaping file: %s: %s" % (str(e), str(traceback.format_exc())))
