#!/usr/bin/env python3
import traceback
import config
import capd
import pickle
import pwd
import grp
import sys
import os
import subprocess
import signal
import MySQLdb as mdb
import logging
import logging.handlers

EXIT_CODE = 0

if os.geteuid() == 0:
    # In testing, we found that this script would produce different results
    # when running as the actual nna user and when running as root with setuid().
    # We're not sure why, but we do need to actually have the nna user run the script.
    # If it runs as root-with-setuid(), the bandwidth graph will stop updating.
    #    -- swolf 2023-12-06
    print("Error: please run this script as the user 'nna'")
    sys.exit(1)

logger = logging.getLogger('nnabackend')
logger.setLevel(logging.DEBUG)

handler = logging.handlers.SysLogHandler()
handler.setLevel(logging.ERROR)
logger.addHandler(handler)

handler = logging.handlers.RotatingFileHandler('/usr/local/nagiosna/var/backend.log', 'a', 1048576, 10)
handler.setLevel(logging.INFO)
logger.addHandler(handler)
        
def start(sourcename):
    try:
        name, directory, port, flowtype, lifetime, id = get_source(sourcename)
    except Exception as e:
        print(str(e))
        raise Exception('Unable to find %s in the database, cannot start it.' % sourcename)
    
    capd.start(name, directory, port, flowtype, lifetime, id)

def start_all():
    try:
        sources = get_sources()
    except Exception as e:
        print('Error connecting to the database. %s' % str(e))
        sys.exit(1)
    
    for source in sources:
        name, directory, port, flowtype, lifetime, id = source
        try:
            capd.start(name, directory, port, flowtype, lifetime, id)
        except ValueError as e:
            print(str(e))
        except Exception as e:
            stop_all()
            raise e

def stop(sourcename):
    try:
        name, directory, port, flowtype, lifetime, id = get_source(sourcename)
    except Exception as e:
        print(str(e))
        raise Exception('Unable to find %s in the database, cannot stop it.' % sourcename)
    
    capd.stop(name, directory, port, flowtype, lifetime, id)

def stop_all():
    try:
        sources = get_sources()
    except Exception as e:
        print('Error connecting to the database. %s' % str(e))
        sys.exit(1)
    
    for name, directory, port, flowtype, lifetime, id in sources:
        try:
            capd.stop(name, directory, port, flowtype, lifetime, id)
        except ValueError as e:
            print(str(e))
        except Exception as e:
            print(str(e))

def status(sourcename):
    _, directory, _, _, _, _ = get_source(sourcename)
    state = False
    check = subprocess.Popen(['ps', '-Afww'], stdout=subprocess.PIPE)
    check.wait()
    for line in check.stdout.readlines():
        if directory in line:
            state = True
    if state:
        print('NagiosNA backend is currently running...')
        sys.exit(200)
    else:
        print('NagiosNA does not appear to be running.')

def get_source(sourcename):
    try:
        source = config.query_database('SELECT `name`, `directory`, `port`, `flowtype`, `lifetime`, `id` FROM sources WHERE `name` = "%s"' % sourcename)
    except Exception as e:
        logger.exception(e)
    return source[0]

def get_sources():
    try:
        sources = config.query_database('SELECT `name`, `directory`, `port`, `flowtype`, `lifetime`, `id` FROM sources')
    except Exception as e:
        logger.exception(e)
        raise e
    return sources

def main():
    if sys.argv[1] == 'start':
        try:
            servicename = sys.argv[2]
            if not servicename:
                raise IndexError
            start(servicename)
            print('%s process started...' % servicename)
        except IndexError:
            start_all()
            print('NagiosNA backend started...')
    elif sys.argv[1] == 'stop':
        try:
            servicename = sys.argv[2]
            if not servicename:
                raise IndexError
            stop(servicename)
            print('%s process stopped.' % servicename)
        except IndexError:
            stop_all()  
            print('NagiosNA backend stopped.')
    elif sys.argv[1] == 'status':
        try:
            servicename = sys.argv[2]
            if not servicename:
                raise IndexError
            status(servicename)
        except IndexError:
            status()

if __name__ == '__main__':
    try:
        main()
    except SystemExit as e:
        sys.exit(int(str(e)))
    except Exception as e:
        print(traceback.print_exc(e))
        print(str(e))
        sys.exit(1)
