#!/bin/bash
# Controller Gate::authorize Checker

APP_PATH=${1:-"/repos/nagios-analyzer-next/nagiosna/app"}
CONTROLLER_DIR="$APP_PATH/Http/Controllers/Api/V1"

# Colors for terminal output
RED="\e[31m"
GREEN="\e[32m"
YELLOW="\e[33m"
ORANGE="\e[34m"
RESET="\e[0m"

echo -e "\nScanning controllers in: $CONTROLLER_DIR\n"

# Counters
total_controllers=0
total_functions=0
missing_auth=0

# Exclude API routes that do not require authorization
# Format: "ControllerName:functionName"
declare -a EXCLUSIONS=(
    "EnvironmentController:get"
    "LicenseController:get"
    "LicenseController:getTrial"
    "InitialInstall:__invoke"
    "ProfileController:updatePersonalInformation"
    "ProfileController:updateLanguage"
    "ProfileController:updateTheme"
    "ProfileController:updatePassword"
    "ProfileController:updateAPIKey"
    "ProfileController:updateDefaultDashboard"
    "AuthController:get"
    "AuthController:installed"
    "AuthController:login"
    "AuthController:logout"
    "AuthController:forgotPassword"
    "AuthController:resetPasswordToken"
    "AuthController:resetPassword"
    "SystemController:getCpuStatus"
    "SystemController:getMemoryStatus"
    "SystemController:getRootDriveStatus"
    "SystemController:getProductInfo"
    "SystemController:getUpdateCheck"
    "SystemController:getTheme"
    "SystemController:getInstalledIntegrations"
    "SystemController:getLanguage"
    "SystemController:getSystemTime"
    "SuricataScanController:isRunning"
    "ExportController:login"
)

is_excluded() {
    local target="$1:$2"
    for exclude in "${EXCLUSIONS[@]}"; do
        if [[ "$exclude" == "$target" ]]; then
            return 0
        fi
    done
    return 1
}

print_func_result() {
    local controller_name="$className"
    local function_name="$func_name"

    if is_excluded "${controller_name/.php/}" "$function_name"; then
        echo -e "    ${YELLOW}✔ $func_name() → EXCLUDED (Authorization not required)${RESET}"
        return
    fi

    if $has_gate; then
        gates_joined=$(IFS=", "; echo "${gate_details[*]}")
        echo -e "    ${GREEN}✔ $func_name() → Gate::authorize($gates_joined)${RESET}"
    else
        echo -e "    ${RED}✘ $func_name() → MISSING Gate::authorize${RESET}"
        ((missing_auth++))
    fi
}

# Iterate through all controllers
while read -r controller; do
    ((total_controllers++))

    className=$(basename "$controller")
    echo -e "${ORANGE}=== $className ===${RESET}"

    inside_func=false
    func_name=""
    has_gate=false
    gate_details=()

    while IFS= read -r line; do

        # Detect PUBLIC functions only
        if [[ $line =~ public[[:space:]]+function[[:space:]]+([a-zA-Z0-9_]+)\( ]]; then
            # Finalize the previous function if we were inside one
            if $inside_func; then
                print_func_result
            fi
            # Start tracking the new public function
            inside_func=true
            func_name="${BASH_REMATCH[1]}"
            # Skip __construct methods - they should not be checked for gates
            if [[ "$func_name" == "__construct" ]]; then
                inside_func=false
                has_gate=false
                gate_details=()
                continue
            fi
            ((total_functions++))
            has_gate=false
            gate_details=()
            continue
        fi

        # Detect PRIVATE / PROTECTED functions → skip them completely
        if [[ $line =~ (private|protected)[[:space:]]+function[[:space:]]+[a-zA-Z0-9_]+\((.*)\) ]]; then
            # If we were tracking a public function, finalize it first
            if $inside_func; then
                print_func_result
            fi
            # Reset state for private/protected methods
            inside_func=false
            has_gate=false
            gate_details=()
            continue
        fi

        # Detect Gate::authorize calls
        if [[ $line =~ Gate::authorize\(.+\) ]]; then
            has_gate=true
            gate_call=$(echo "$line" | grep -o "Gate::authorize([^)]*)" | sed 's/Gate::authorize(//;s/)//')
            gate_details+=("$gate_call")
        fi

    done < "$controller"

    # Finalize the LAST function in the file
    if $inside_func; then
        print_func_result
    fi

    echo ""
done < <(find "$CONTROLLER_DIR" -type f -name "*.php")

echo -e "\n${YELLOW}Summary:${RESET}"
echo -e "   Controllers scanned: ${GREEN}$total_controllers${RESET}"
echo -e "   Public functions: ${GREEN}$total_functions${RESET}"
echo -e "   Missing Gate::authorize: ${RED}$missing_auth${RESET}"

if [[ $missing_auth -gt 0 ]]; then
    echo -e "\n${RED}ERROR: $missing_auth functions are missing Gate::authorize!${RESET}"
    exit 1
else
    echo -e "\n${GREEN}SUCCESS: All public functions are authorized or excluded. :)${RESET}"
    exit 0
fi
