<?php

namespace App\Jobs\Middleware;

use Closure;
use App\Services\LicenseService;
use App\Services\TrialService;
use App\Services\MaintenanceService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use App\Services\SuricataService;

/**
 * Job middleware to prevent jobs from running when license/trial/maintenance is invalid.
 * 
 * Jobs are blocked when:
 * - Trial license is expired (type=trial && trial_expired)
 * - Licensed but not activated (type=licensed && trial_expired && !activated)
 * - Licensed but maintenance expired (type=licensed && trial_expired && maintenance_expired)
 * 
 * Usage:
 *   // Always check license
 *   new CheckLicenseStatus()
 * 
 *   // Only check license for specific check types (for CheckJob)
 *   new CheckLicenseStatus(onlyForCheckTypes: ['nmap', 'suricata'])
 * 
 * @see \App\Policies\BasePolicy::before() - Same logic for API requests
 * @see /nagiosna/ui/src/components/RouteProtector.tsx - Same logic for frontend routes
 */
class CheckLicenseStatus
{
    /**
     * @param array<string>|null $checkTypes If provided, only run license check for these check_type values.
     *                                        Only applies to jobs with a `check` property that has a `check_type`.
     */
    public function __construct(
        protected ?array $checkTypes = null
    ) {}

    /**
     * Process the queued job.
     *
     * @param  mixed  $job
     * @param  Closure  $next
     * @return void
     */
    public function handle(object $job, Closure $next): void
    {
        if ($this->checkTypes !== null) {
            $checkType = $this->getCheckTypeFromJob($job);
            
            if ($checkType === null || !in_array($checkType, $this->checkTypes)) {
                $next($job);
                return;
            }
        }

        $licenseStatus = Cache::remember('job_license_status', 60, function () {
            return $this->checkLicenseStatus();
        });

        if ($licenseStatus['should_block']) {
            Log::info('Job skipped - license/maintenance issue', [
                'job' => get_class($job),
                'reason' => $licenseStatus['reason'],
            ]);

            // Stop Suricata scan if it's running to prevent disk from filling up
            $suricata = app(SuricataService::class);
            $suricata->stopSuricataScan($licenseStatus['reason']);

            return;
        }

        $next($job);
    }

    /**
     * Extract check_type from a job if it has a check property.
     */
    protected function getCheckTypeFromJob(object $job): ?string
    {
        if (property_exists($job, 'check') && isset($job->check->check_type)) {
            return $job->check->check_type;
        }
        
        return null;
    }

    /**
     * Check the license, trial, and maintenance status.
     * 
     * @return array{should_block: bool, reason: string|null}
     */
    protected function checkLicenseStatus(): array
    {
        try {
            $license = app(LicenseService::class);
            $trial = app(TrialService::class);
            $maintenance = app(MaintenanceService::class);

            $trialExpired = $license->isTrialLicense() && $trial->isExpired();
            $notActivated = !$license->isTrialLicense() && $trial->isExpired() && !$license->isActivated();
            $maintenanceExpired = !$license->isTrialLicense() && $trial->isExpired() && $maintenance->isExpired();

            $shouldBlock = $trialExpired || $notActivated || $maintenanceExpired;

            $reason = null;
            if ($trialExpired) {
                $reason = 'trial_expired';
            } elseif ($notActivated) {
                $reason = 'not_activated';
            } elseif ($maintenanceExpired) {
                $reason = 'maintenance_expired';
            }

            return [
                'should_block' => $shouldBlock,
                'reason' => $reason,
            ];
        } catch (\Exception $e) {
            Log::warning('Could not check license status for queue job', [
                'error' => $e->getMessage()
            ]);

            return ['should_block' => false, 'reason' => null];
        }
    }
}
