/*
NOTE:
This file uses puppeteer which requires node.
In order to avoid installing node and puppeteer on users' machines, it is bundled into a single executable.
It is first bundled with puppeteer using esbuild (https://esbuild.github.io) then bundled with node using SEA (https://nodejs.org/api/single-executable-applications.html).
Whenever you make changes, run the following (things may have changed, you should be able to figure it out).
Prerequisites:
- Install node (https://nodejs.org/en/download)
- Install the correct version of chrome-for-testing (not required for bundling, but required for testing)
  - As of creating this executable, this version of puppeteer uses chrome 140.0.7339.82 - https://storage.googleapis.com/chrome-for-testing-public/140.0.7339.82/linux64/chrome-linux64.zip
  - If remaking the executable, check the version of chrome it uses - https://github.com/puppeteer/puppeteer/blob/main/packages/puppeteer-core/src/revisions.ts
  - Doing it this way will keep this script + the version of chrome installed static and, in their words, "guaranteed to work"
Esbuild:
- Install esbuild: npm install --save-exact --save-dev esbuild
- Check it's installed: ./node_modules/.bin/esbuild --version
- Bundle the file: ./node_modules/.bin/esbuild export.js --bundle --outfile=out.cjs --platform=node --external:typescript
SEA:
- Create config file: echo '{ "main": "out.cjs", "output": "sea-prep.blob" }' > sea-config.json
- Generate blob: node --experimental-sea-config sea-config.json
- Copy the node executable: cp $(command -v node) out
- Inject the blob:
  npx postject out NODE_SEA_BLOB sea-prep.blob \
    --sentinel-fuse NODE_SEA_FUSE_fce680ab2cc467b6e072b8b5df1996b2
Postrequisites:
  - Make sure the bundled file is in /repos/nagios-analyzer-next/nagiosna/resources/js/
  - Rename the file to export
*/

import puppeteer from 'puppeteer-core';
// Or import puppeteer from 'puppeteer-core';

async function main() {
  const args = process.argv;
  const token = args[2];
  const id = args[3];
  const type = args[4];
  const path = args[5];

  try {
    // Launch the browser and open a new blank page
    const browser = await puppeteer.launch({
      args: ['--no-sandbox', '--disable-setuid-sandbox'],
      executablePath: '/var/www/html/nagiosna/exporting/chrome-linux64/chrome',
      userDataDir: '/dev/null'
    });

    // Fetch the token data
    const response = await fetch(`http://localhost/api/v1/export/login`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      },
    });
    const cookies = response.headers.get('set-cookie');
    const cookieArray = cookies.split(',');
    const sessionCookie = cookieArray.find(cookie => cookie.trim().startsWith('laravel_session='));
    const sessionValue = sessionCookie.split('=')[1].split(';')[0];

    // Set the session cookie for Puppeteer
    await browser.setCookie(
      {
        name: 'laravel_session',
        value: sessionValue,
        domain: 'localhost',
        path: '/',
        httpOnly: true,
        secure: false
      },
    );

    // Open a new page
    const page = await browser.newPage();

    // Set the URL for the page based on input
    const url = `http://localhost/dashboards/${id}?export=true`;
    
    // Navigate to the URL
    await page.goto(url, {
      waitUntil: 'networkidle0'
    });

    // Depending on the type, either save a PDF or screenshot
    if (type === 'pdf') {
      await page.pdf({
        path: path,
        format: 'A4',
        landscape: true,
        scale: 0.75,
        printBackground: true
      });
    } else if (type === 'jpg') {
      await page.setViewport({ width: 1500, height: 1050 });
      await page.screenshot({
        path: path,
        type: 'jpeg'
      });
    }

    // Close the browser
    await browser.close();
  } catch (error) {
    console.error('Error occurred:', error);
  }
}

// Call the main function
main();
