import {
  LayoutDashboard,
  CircleAlert,
  FileText,
  HelpCircle,
  Layers,
  Shield,
  Network,
  ChevronDown,
  GlobeLockIcon,
  Radar,
  ChevronsUpDown,
  CircleUser,
  LogOut,
} from 'lucide-react';
import { Link, useLocation, useNavigate } from 'react-router';
import { useTranslation } from 'react-i18next';
import { useMemo, useState } from 'react';
import NagiosIcon from '@/assets/images/nagios-n.svg';
import { useAuth } from '@/contexts/AuthContext';
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarHeader,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarFooter,
  SidebarMenuSub,
  SidebarMenuSubItem,
  SidebarMenuSubButton,
  useSidebar,
} from '@/components/ui/sidebar';
import { Collapsible, CollapsibleTrigger, CollapsibleContent } from '@/components/ui/collapsible';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import axios from '@/lib/axios';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import SharkFin from '@/components/ui/icons/SharkFin';
import { useGetProductInfo } from '@/components/queries';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { isAdmin, hasNoViewPermissions } from '@/components/role/utils';

// Menu items based on your routes
const items = [
  {
    title: 'Flow Sources',
    path: '/sources',
    icon: Network,
    key: 'flow_source',
  },
  {
    title: 'Groups',
    path: '/groups',
    icon: Layers,
    key: 'flow_source',
  },
  {
    title: 'Reports',
    path: '/reports',
    icon: FileText,
  },
  {
    title: 'Alerting',
    path: '/alerting',
    icon: CircleAlert,
  },
  {
    title: 'Wireshark',
    path: '/wireshark',
    icon: SharkFin,
    key: 'wireshark',
    translate: false,
  },
  {
    title: 'Nmap',
    path: '/nmap',
    icon: Radar,
    key: 'nmap',
    translate: false,
  },
  {
    title: 'Suricata',
    path: '/suricata',
    icon: GlobeLockIcon,
    key: 'suricata',
    translate: false,
  },
  {
    title: 'Help',
    path: '/help',
    icon: HelpCircle,
  },
];

const adminItems = [
  {
    title: 'Global Settings',
    path: '/admin/globals',
  },
  {
    title: 'Notification Settings',
    path: '/admin/notifications',
  },
  {
    title: 'License Information',
    path: '/admin/license',
  },
  {
    title: 'User Management',
    path: '/admin/users',
  },
  {
    title: 'Roles',
    path: '/admin/roles',
  },
  {
    title: 'LDAP/AD Integration',
    path: '/admin/auth_servers',
  },
  {
    title: 'System Backup',
    path: '/admin/backup',
  },
];

export function AppSidebar() {
  const location = useLocation();
  const { t } = useTranslation();
  const navigate = useNavigate();
  const { user } = useAuth();
  const queryClient = useQueryClient();
  const { data: productInfo } = useGetProductInfo();
  const { state } = useSidebar();
  const [adminDropdownOpen, setAdminDropdownOpen] = useState(false);

  const visibleItems = useMemo(() => {
    return items.filter((item) => {
      if (!item.key) return true;
      return !hasNoViewPermissions(user?.role, item.key);
    });
  }, [user?.role]);

  const logoutMutation = useMutation({
    mutationFn: () => {
      return axios.post('/api/v1/logout');
    },
    onSuccess: async () => {
      await queryClient.invalidateQueries({ queryKey: ['auth'] });
      localStorage.removeItem('login_locale');
      localStorage.removeItem('locale');
      navigate('/login');
    },
  });

  return (
    <Sidebar collapsible="icon">
      <SidebarHeader>
        <SidebarMenu>
          <SidebarMenuItem>
            <SidebarMenuButton asChild className="mt-0.5 py-5">
              <Link to="/">
                <img src={NagiosIcon} alt="Nagios NA" className="!size-6" />
                <span className="text-lg font-semibold">Network Analyzer</span>
              </Link>
            </SidebarMenuButton>
          </SidebarMenuItem>
        </SidebarMenu>
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupContent>
            <SidebarMenu>
              {user && (
                <SidebarMenuItem key={'dashboards'}>
                  <SidebarMenuButton asChild isActive={location.pathname.slice(0, 11) === '/dashboards'}>
                    <Link to={'/dashboards/' + user.default_dashboard}>
                      <LayoutDashboard />
                      <span>{t('Dashboards')}</span>
                    </Link>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              )}
              {visibleItems.map((item) => {
                return (
                  <SidebarMenuItem key={item.title}>
                    <SidebarMenuButton asChild isActive={location.pathname === item.path}>
                      <Link to={item.path}>
                        <item.icon />
                        <span>{item.translate === false ? item.title : t(item.title)}</span>
                      </Link>
                    </SidebarMenuButton>
                  </SidebarMenuItem>
                );
              })}
              {isAdmin(user?.role) &&
                (state === 'expanded' ? (
                  <Collapsible className="group/collapsible">
                    <SidebarMenuItem>
                      <CollapsibleTrigger asChild>
                        <SidebarMenuButton>
                          <Shield />
                          <span>{t('Administration')}</span>
                          <ChevronDown className="ml-auto transition-transform group-data-[state=open]/collapsible:rotate-180" />
                        </SidebarMenuButton>
                      </CollapsibleTrigger>
                      <CollapsibleContent>
                        <SidebarMenuSub>
                          {adminItems.map((item) => (
                            <SidebarMenuSubItem key={item.title}>
                              <SidebarMenuSubButton isActive={location.pathname === item.path} asChild>
                                <Link to={item.path}>
                                  <span>{t(item.title)}</span>
                                </Link>
                              </SidebarMenuSubButton>
                            </SidebarMenuSubItem>
                          ))}
                        </SidebarMenuSub>
                      </CollapsibleContent>
                    </SidebarMenuItem>
                  </Collapsible>
                ) : (
                  <SidebarMenuItem>
                    <DropdownMenu open={adminDropdownOpen} onOpenChange={setAdminDropdownOpen}>
                      <DropdownMenuTrigger asChild>
                        <SidebarMenuButton>
                          <Shield />
                          <span>{t('Administration')}</span>
                        </SidebarMenuButton>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent side="right" align="start" onMouseLeave={() => setAdminDropdownOpen(false)}>
                        {adminItems.map((item) => {
                          const isActive = location.pathname === item.path;
                          return (
                            <DropdownMenuItem
                              key={item.title}
                              className={isActive ? 'bg-accent text-accent-foreground' : ''}
                              onClick={(e) => {
                                e.preventDefault();
                                navigate(item.path);
                              }}
                            >
                              {item.title}
                            </DropdownMenuItem>
                          );
                        })}
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </SidebarMenuItem>
                ))}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
      <SidebarFooter>
        <SidebarMenu>
          <SidebarMenuItem>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <SidebarMenuButton size="lg" className="group/profile">
                  <Avatar>
                    <AvatarImage src={`/storage/avatars/${user?.username}.svg`} className="scale-95" />
                    <AvatarFallback>{user?.username.slice(0, 2).toUpperCase()}</AvatarFallback>
                    <div className="to-background/50 dark:to-foreground/50 absolute size-full -translate-x-6 translate-y-6 rotate-45 bg-linear-to-t from-transparent delay-100 group-hover/profile:translate-x-6 group-hover/profile:-translate-y-6 group-hover/profile:transition-transform group-hover/profile:duration-700 group-data-[state=open]/profile:translate-x-6 group-data-[state=open]/profile:-translate-y-6" />
                  </Avatar>
                  <div className="flex flex-col">
                    {user?.username || t('Username')}
                    {productInfo?.version && (
                      <span className="text-muted-foreground block text-xs">{productInfo.version}</span>
                    )}
                  </div>
                  <ChevronsUpDown className="ml-auto" />
                </SidebarMenuButton>
              </DropdownMenuTrigger>
              <DropdownMenuContent className="w-(--radix-dropdown-menu-trigger-width)">
                <Link to="/profile">
                  <DropdownMenuItem>
                    <CircleUser />
                    <span>{t('Profile')}</span>
                  </DropdownMenuItem>
                </Link>
                <DropdownMenuItem onClick={() => logoutMutation.mutate()}>
                  <LogOut />
                  <span>{t('Log out')}</span>
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </SidebarMenuItem>
        </SidebarMenu>
      </SidebarFooter>
    </Sidebar>
  );
}
