import { AlertCircle, Clock } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { Link } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';
import { Badge } from '@/components/ui/badge';
import { isAdmin } from '@/components/role/utils';

type BannerVariant = 'red' | 'yellow';
type BannerSeverity = 'Critical' | 'Warning' | 'Notice';

export function Banner() {
  const { t } = useTranslation();
  const { user, license } = useAuth();

  if (!license || !user) return null;

  const userIsAdmin = isAdmin(user.role);

  let text = '';
  let variant: BannerVariant = 'red';
  let severity: BannerSeverity = 'Critical';
  let icon: React.ReactNode = <AlertCircle className="h-3 w-3" />;
  let shouldShow = false;
  let isMaintenanceBanner = false;

  // License not activated
  if (!license.activated) {
    text = t(
      'Your license has not yet been activated. Access to certain features of Nagios Network Analyzer are restricted or limited.'
    );
    variant = 'red';
    severity = 'Critical';
    icon = <AlertCircle className="h-3 w-3" />;
    shouldShow = true;
  }

  // Maintenance expired
  else if (license.maintenance_expired || license.maintenance_days_left <= 0) {
    text = t('Your maintenance has expired. Click here to renew your maintenance.');
    variant = 'red';
    severity = 'Warning';
    icon = <AlertCircle className="h-3 w-3" />;
    shouldShow = true;
    isMaintenanceBanner = true;
  }

  // Maintenance expiring in less than 30 days
  else if (!license.maintenance_expired && license.maintenance_days_left < 30 && license.maintenance_days_left > 0) {
    text = t('Your maintenance expires in {{days}} days. Click to here to renew your maintenance.', {
      days: license.maintenance_days_left,
    });
    variant = 'yellow';
    severity = 'Notice';
    icon = <Clock className="h-3 w-3" />;
    shouldShow = true;
    isMaintenanceBanner = true;
  }

  // Trial expiring in less than 10 days
  else if (
    license.type === 'trial' &&
    !license.trial_expired &&
    license.trial_days_left <= 10 &&
    license.trial_days_left >= 0
  ) {
    text = t('Your trial expires in {{days}} days. Please activate a full license to continue using all features.', {
      days: license.trial_days_left,
    });
    variant = license.trial_days_left <= 3 ? 'red' : 'yellow';
    severity = 'Notice';
    icon = <Clock className="h-3 w-3" />;
    shouldShow = true;
  }

  if (!shouldShow) return null;

  return (
    <div className="bg-popover flex w-full items-center justify-center gap-2 pt-2">
      <Badge variant={variant} className="flex items-center gap-1">
        {icon}
        {t(severity)}
      </Badge>

      {isMaintenanceBanner ? (
        <a
          href="https://www.nagios.com/renew/"
          target="_blank"
          rel="noopener noreferrer"
          className={`hover:underline text-badge-${variant}-text`}
        >
          {text}
        </a>
      ) : userIsAdmin ? (
        <Link to="/admin/license" className={`hover:underline text-badge-${variant}-text`}>
          {text}
        </Link>
      ) : (
        <span className={`text-badge-${variant}-text`}>{text}</span>
      )}
    </div>
  );
}
