import { Navigate, Outlet, useLocation } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';
import { isAdmin } from './role/utils';

export default function RouteProtector() {
  const { user, installed, isUserLoading, isInstalledLoading, license, isLicenseLoading } = useAuth();

  const location = useLocation();
  const path = location.pathname;

  if (isUserLoading || isInstalledLoading || isLicenseLoading || !license) return null;

  let type: 'login' | 'install' | 'admin' | 'activate' | 'integration' | 'protected' = 'protected';

  if (path.startsWith('/login') || path.startsWith('/forgot_password') || path.startsWith('/reset_password'))
    type = 'login';
  else if (path.startsWith('/install')) type = 'install';
  else if (path.startsWith('/activate')) type = 'activate';
  else if (path.startsWith('/nmap') || path.startsWith('/wireshark') || path.startsWith('/suricata'))
    type = 'integration';
  else if (path.startsWith('/admin')) type = 'admin';

  // If not authenticated, redirect to login
  if (!user && (type === 'protected' || type === 'admin' || type === 'integration')) {
    return <Navigate to="/login" replace />;
  }

  if (!installed && type !== 'install') return <Navigate to="/install" replace />;
  if (installed && type === 'install') return <Navigate to="/login" replace />;

  // Redirect /login to main if already logged in
  if (type === 'login' && user) {
    return <Navigate to={`/`} replace />;
  }

  // Redirect integration pages to /maintenance page if maintenance is expired
  if (type === 'integration') {
    if (license.maintenance_expired) {
      return <Navigate to="/maintenance" replace />;
    }
  }

  // Redirect non-admins to login or 404 for admin routes
  if (type === 'admin') {
    if (!user) return <Navigate to="/login" replace />;
    if (!isAdmin(user?.role)) return <Navigate to="/404" replace />;
  }

  // Redirect /activate to main if trial is not expired OR we have a license key and it's activated
  if (type === 'activate' && (!license.trial_expired || (license.type === 'licensed' && license.activated))) {
    return <Navigate to="/" replace />;
  }

  // Redirect to /activate page if trial is expired and no license key OR trial is expired and license key exists but is not activated
  if (
    (type === 'protected' || type === 'admin' || type === 'integration') &&
    ((license.trial_expired && license.type !== 'licensed') ||
      (license.trial_expired && license.type === 'licensed' && !license.activated))
  ) {
    return <Navigate to={`/activate`} replace />;
  }

  return <Outlet />;
}
