import { Input } from '@/components/ui/input';
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem, SelectGroup } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '../ui/table';
import { ScrollArea } from '../ui/scroll-area';
import { useFieldArray } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { UseFormReturn } from 'react-hook-form';
import { ImportUser } from '@/components/administration/types';
import { Shield, User } from 'lucide-react';
import { useGetRoles } from '@/components/role/queries';

interface ImportUserStep3Props {
  formImport: UseFormReturn<{ formUsers: ImportUser[] }>;
  formUsers: ImportUser[];
  selectedIndices: number[];
  setSelectedIndices: React.Dispatch<React.SetStateAction<number[]>>;
}

export default function ImportUserStep3({
  formImport,
  formUsers,
  selectedIndices,
  setSelectedIndices,
}: ImportUserStep3Props) {
  const { t } = useTranslation();
  const { control } = formImport;
  const { fields, update } = useFieldArray({
    control,
    name: 'formUsers',
  });
  const { data: roles } = useGetRoles();

  if (!roles) return;

  return (
    <>
      <div className="bg-card flex items-center justify-end border-b px-4 pb-4">
        <div className="text-muted-foreground flex items-center gap-4 text-sm">
          <div>
            {t('Selected Users:')} {selectedIndices.length}
          </div>
          <div>
            <span className="bg-success inline-block h-2 w-2 rounded-full"></span>{' '}
            {
              selectedIndices.filter(
                (index) =>
                  formImport.watch(`formUsers.${index}.username`)?.trim() &&
                  formImport.watch(`formUsers.${index}.email`)?.trim()
              ).length
            }
          </div>
          <div>
            <span className="bg-error inline-block h-2 w-2 rounded-full"></span>{' '}
            {
              selectedIndices.filter(
                (index) =>
                  !formImport.watch(`formUsers.${index}.username`)?.trim() ||
                  !formImport.watch(`formUsers.${index}.email`)?.trim()
              ).length
            }
          </div>
        </div>
      </div>

      <ScrollArea className="h-130">
        <Table>
          <TableHeader className="bg-background sticky top-0 z-10 p-4">
            <TableRow>
              <TableHead className="text-muted-foreground">
                <Checkbox
                  checked={selectedIndices.length === formUsers.length}
                  onCheckedChange={(checked) => {
                    if (checked) {
                      setSelectedIndices(formUsers.map((_, index) => index));
                    } else {
                      setSelectedIndices([]);
                    }
                  }}
                />
              </TableHead>
              <TableHead className="text-muted-foreground">{t('Display Name')}</TableHead>
              <TableHead className="text-muted-foreground">
                {t('Username')} <span className="text-error">*</span>
              </TableHead>
              <TableHead className="text-muted-foreground">
                {t('Email')} <span className="text-error">*</span>
              </TableHead>
              <TableHead className="text-muted-foreground">
                {t('User Type')} <span className="text-error">*</span>
              </TableHead>
              <TableHead className="text-muted-foreground">{t('API Access')}</TableHead>
              <TableHead className="text-muted-foreground">{t('Auth Type')}</TableHead>
              <TableHead className="text-muted-foreground">{t('Status')}</TableHead>
            </TableRow>
          </TableHeader>

          <TableBody>
            {fields.map((field, index) => {
              const username = formImport.watch(`formUsers.${index}.username`)?.trim();
              const email = formImport.watch(`formUsers.${index}.email`)?.trim();
              const isValidEmail = !!email && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);

              return (
                <TableRow key={field.id}>
                  <TableCell className="w-[50px]">
                    <Checkbox
                      checked={selectedIndices.includes(index)}
                      onCheckedChange={(checked) => {
                        if (checked) {
                          setSelectedIndices((prev) => [...prev, index]);
                        } else {
                          setSelectedIndices((prev) => prev.filter((i) => i !== index));
                        }
                      }}
                    />
                  </TableCell>
                  <TableCell className="w-[300px]">
                    <Input
                      {...formImport.register(`formUsers.${index}.displayname`, { required: false })}
                      className={
                        formImport.watch(`formUsers.${index}.displayname`) ? 'border-primary' : 'border-red-500'
                      }
                    />
                  </TableCell>
                  <TableCell className="w-[300px]">
                    <Input
                      {...formImport.register(`formUsers.${index}.username`, { required: true })}
                      className={username ? 'border-primary' : 'border-red-500'}
                    />
                  </TableCell>
                  <TableCell className="w-[300px]">
                    <Input
                      {...formImport.register(`formUsers.${index}.email`, { required: true })}
                      className={isValidEmail ? 'border-primary' : 'border-red-500'}
                    />
                  </TableCell>
                  <TableCell className="w-[150px]">
                    <Select
                      value={`${field.role_id}`}
                      onValueChange={(val) => {
                        const currentValues = formImport.getValues(`formUsers.${index}`);
                        update(index, { ...currentValues, role_id: Number(val) });
                      }}
                    >
                      <SelectTrigger className="w-xs">
                        <SelectValue placeholder={t('Select a role')} />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectGroup>
                          {roles.map((role) => (
                            <SelectItem
                              key={role.id}
                              value={`${role.id}`}
                              className="hover:bg-secondary hover:cursor-pointer"
                            >
                              {role.type === 'admin' && <Shield />}
                              {role.type === 'user' && <User />}
                              {role.name}
                            </SelectItem>
                          ))}
                        </SelectGroup>
                      </SelectContent>
                    </Select>
                  </TableCell>
                  <TableCell className="w-[150px]">
                    <Select
                      value={String(field.apiaccess)}
                      onValueChange={(val) => {
                        const currentValues = formImport.getValues(`formUsers.${index}`);
                        update(index, { ...currentValues, apiaccess: val === 'true' });
                      }}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="false">{t('No')}</SelectItem>
                        <SelectItem value="true">{t('Yes')}</SelectItem>
                      </SelectContent>
                    </Select>
                  </TableCell>
                  <TableCell className="text-muted-foreground w-[100px]">{field.type}</TableCell>
                  <TableCell className="w-[150px]">
                    {username && isValidEmail ? (
                      <div className="flex items-center gap-2">
                        <span className="bg-success inline-block h-2 w-2 rounded-full"></span>
                        {t('Ready')}
                      </div>
                    ) : (
                      <div className="flex items-center gap-2">
                        <span className="bg-error inline-block h-2 w-2 rounded-full"></span>
                        {t('Incomplete')}
                      </div>
                    )}
                  </TableCell>
                </TableRow>
              );
            })}
          </TableBody>
        </Table>
      </ScrollArea>
    </>
  );
}
