import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useTranslation } from 'react-i18next';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form';
import { Card, CardContent } from '@/components/ui/card';
import { PageSeparator } from '@/components/Page';
import { useGetLicenseKey, useSetLicenseKey, useSetLicenseType } from '@/components/administration/license/queries';
import type { License } from '@/components/administration/license/types';
import { useAuth } from '@/contexts/AuthContext';
import { Badge } from '@/components/ui/badge';
import { obfuscateKey } from '@/components/administration/license/utils';
import { useState } from 'react';

export default function LicenseForm() {
  const { t } = useTranslation();
  const setLicenseKey = useSetLicenseKey();
  const setLicenseType = useSetLicenseType();
  const { license } = useAuth();
  const { data: licenseKey } = useGetLicenseKey();

  const originalKey = licenseKey ?? '';
  const [hasUserModified, setHasUserModified] = useState(false);

  const form = useForm<{ key: string }>({
    values: {
      key: originalKey ?? '',
    },
  });

  const { isDirty } = form.formState;
  const currentKeyValue = form.watch('key') ?? '';

  const onSubmit: SubmitHandler<{ key: string }> = (data) => {
    setLicenseKey.mutate(data.key ?? '');
  };

  const updateLicenseType = (type: License['type']) => {
    if (license?.type === 'licensed' || licenseKey == '') return;
    setLicenseType.mutate({ type });
  };

  let formattedDate = '';
  if (license) {
    const maintenanceEndDate = new Date();
    maintenanceEndDate.setDate(maintenanceEndDate.getDate() + license.maintenance_days_left);
    formattedDate = maintenanceEndDate.toLocaleDateString();
  }

  return (
    <Card
      className={`w-1/2 ${licenseKey != null ? 'cursor-pointer' : ''} ${license?.maintenance_expired && license?.type === 'licensed' ? 'border-badge-red-border' : license?.type === 'licensed' ? 'border-badge-green-border' : ''} `}
      onMouseDown={() => {
        updateLicenseType('licensed');
      }}
    >
      <CardContent>
        <div className="flex flex-row items-center justify-between">
          <div className="flex w-full flex-row items-center justify-between">
            <div className="flex items-center">
              <h1 className="text-md font-large leading-9 font-semibold">{t('Licensed')}</h1>
              {license && !license.maintenance_expired && license.type === 'licensed' && (
                <Badge
                  className="ml-2"
                  variant={
                    license.maintenance_days_left >= 180
                      ? 'green'
                      : license.maintenance_days_left >= 30
                        ? 'yellow'
                        : 'red'
                  }
                >
                  {t('Maintenance expires in ')} {license.maintenance_days_left} {t('days on')} {formattedDate}
                </Badge>
              )}
              {license && license.maintenance_expired && (
                <Badge className="ml-2" variant={'red'}>
                  {t('Maintenance expired')}
                </Badge>
              )}
            </div>
            <Button
              disabled={!isDirty}
              variant={isDirty ? 'default' : 'outline'}
              className="border"
              type="submit"
              form="license-form"
            >
              {t('Update')}
            </Button>
          </div>
        </div>

        <PageSeparator />

        <div onMouseDown={(e) => e.stopPropagation()}>
          <Form {...form}>
            <form id="license-form" onSubmit={form.handleSubmit(onSubmit)}>
              <FormField
                control={form.control}
                name="key"
                render={({ field }) => (
                  <FormItem className="flex flex-wrap items-center justify-between">
                    <FormLabel>{t('License Key')}</FormLabel>
                    <FormControl className="w-md">
                      <Input
                        {...field}
                        value={hasUserModified ? currentKeyValue : obfuscateKey(originalKey ?? '')}
                        onChange={(e) => {
                          const newValue = e.target.value;
                          field.onChange(newValue);
                          setHasUserModified(newValue !== originalKey);
                        }}
                        placeholder="XXXXXX-XXXXXX-XXXXXX-XXXXXX-XXXXXX"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />
            </form>
          </Form>
        </div>
      </CardContent>
    </Card>
  );
}
