import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { NotificationOptions } from '@/components/administration/types';
import { toast } from 'sonner';
import { CircleAlert } from 'lucide-react';

export function useGetNotificationOptions() {
  return useQuery<NotificationOptions>({
    queryKey: ['notification_options'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/administration/notifications');
      return response.data;
    },
  });
}

export function useEditNotificationOptionsMutation() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: NotificationOptions) => {
      return axios.put('/api/v1/administration/notifications', {
        notification_channel: data.notification_channel,
        mail_config: data.mail_config,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['notification_options'] });
      toast(t('Notification settings updated.'));
    },
    onError: () => {
      toast(t('Failed to update notification settings.'), { description: t('Please try again.') });
    },
  });
}

export function useTestNotificationOptions() {
  const { t } = useTranslation();

  return useMutation({
    mutationFn: () => {
      return axios.post('/api/v1/administration/notifications', {
        test: 'test',
      });
    },
    onSuccess: async (response) => {
      toast(t('Test email sent'), { description: response.data.message });
    },
    onError: (error: AxiosError<{ message: string }>) => {
      toast(
        <div>
          <div className="flex items-center gap-2">
            <CircleAlert color={'var(--background)'} fill={'var(--foreground)'} className="h-5 w-5" />
            {t('Failed to send test email')}:
          </div>
          <div className="text-muted-foreground mt-2 max-h-50 overflow-auto">
            {error.response?.data.message ?? t('Please try again')}
          </div>
        </div>
      );
    },
  });
}
