import { useState, useMemo } from 'react';
import { AlertingType, AlertRecipient } from '@/components/alerting/types';
import { UseFormReturn, FieldValues, Path } from 'react-hook-form';
import { FormItem, FormControl, FormLabel, FormField } from '@/components/ui/form';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Loader2Icon } from 'lucide-react';
import { useGetUsers } from '../user/queries';
import { useGetServiceHostnames, useGetSNMPReceivers, useGetCommands } from './queries';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Card, CardContent } from '@/components/ui/card';
import { PageSeparator } from '@/components/Page';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import { Button } from '@/components/ui/button';
import { useTranslation } from 'react-i18next';

export default function CheckReceiverSelector<T extends FieldValues>({
  alertingType,
  form,
  step,
}: {
  alertingType: AlertingType;
  form: UseFormReturn<T>;
  step: 'step2' | 'step3';
}) {
  const { t } = useTranslation();
  const [searchQuery, setSearchQuery] = useState('');
  const { data: users, isLoading: usersLoading } = useGetUsers();
  const { data: nagiosRecipients, isLoading: nagiosRecipientsLoading } = useGetServiceHostnames();
  const { data: snmpRecipients, isLoading: snmpRecipientsLoading } = useGetSNMPReceivers();
  const { data: commandRecipients, isLoading: commandRecipientsLoading } = useGetCommands();
  const isLoading = usersLoading || nagiosRecipientsLoading || snmpRecipientsLoading || commandRecipientsLoading;

  const recipientTypeText =
    alertingType === 'user'
      ? 'users'
      : alertingType === 'nagios'
        ? 'service hostnames'
        : alertingType === 'snmp_receiver'
          ? 'SNMP receivers'
          : alertingType === 'command'
            ? 'commands'
            : 'receivers';

  function getRecipientLabelText(recipient: AlertRecipient, alertingType: AlertingType): string {
    if (
      alertingType === 'nagios' &&
      'hostname' in recipient &&
      'servicename' in recipient &&
      'nagios_server' in recipient
    ) {
      return `${recipient.hostname}/${recipient.servicename} on ${recipient.nagios_server.name}`;
    } else if (alertingType === 'user' && 'username' in recipient) {
      return recipient.username;
    } else if (alertingType === 'snmp_receiver' && 'name' in recipient) {
      return recipient.name;
    } else if (alertingType === 'command' && 'name' in recipient) {
      return recipient.name;
    }
    return 'unknown';
  }

  const allRecipients: AlertRecipient[] = useMemo(() => {
    return alertingType === 'user'
      ? (users ?? [])
      : alertingType === 'nagios'
        ? (nagiosRecipients ?? [])
        : alertingType === 'snmp_receiver'
          ? (snmpRecipients ?? [])
          : alertingType === 'command'
            ? (commandRecipients ?? [])
            : [];
  }, [alertingType, users, nagiosRecipients, snmpRecipients, commandRecipients]);

  const filteredRecipients = useMemo(() => {
    const sorted = [...allRecipients].sort((a, b) => {
      const labelA = getRecipientLabelText(a, alertingType).toLowerCase();
      const labelB = getRecipientLabelText(b, alertingType).toLowerCase();
      return labelA.localeCompare(labelB);
    });

    if (!searchQuery.trim()) {
      return sorted;
    }

    const query = searchQuery.toLowerCase().trim();
    return sorted.filter((recipient: AlertRecipient) => {
      const label = getRecipientLabelText(recipient, alertingType);
      return label?.toLowerCase().includes(query) ?? false;
    });
  }, [allRecipients, searchQuery, alertingType]);

  const handleSelectUnselectAll = () => {
    const fieldPath = `${step}.${alertingType}` as Path<T>;
    const currentSelection = (form.getValues(fieldPath) as number[] | undefined) || [];
    const filteredIds = filteredRecipients.map((recipient: AlertRecipient) => recipient.id);
    const allSelected = filteredIds.every((id) => currentSelection.includes(id));

    let newSelection: number[];
    if (allSelected) {
      newSelection = currentSelection.filter((id) => !filteredIds.includes(id));
    } else {
      newSelection = Array.from(new Set([...currentSelection, ...filteredIds]));
    }

    form.setValue(fieldPath, newSelection as T[Path<T>]);
  };

  if (isLoading) {
    return (
      <Card>
        <CardContent>
          <div className="flex h-64 items-center justify-center">
            <Loader2Icon className="text-muted-foreground h-6 w-6 animate-spin" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardContent>
        <div className="flex items-center gap-2">
          <Input
            type="text"
            placeholder={t(`Search ${recipientTypeText}...`)}
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="flex-1"
          />
          <Button size="sm" variant="outline" type="button" onClick={() => handleSelectUnselectAll()}>
            {t('Select/Unselect All')}
          </Button>
        </div>
        <PageSeparator />
        <ScrollArea className="-mr-4 h-[300px] pr-4">
          {filteredRecipients.length > 0 ? (
            <div className="grid grid-cols-1 gap-3 sm:grid-cols-2 md:grid-cols-3">
              {filteredRecipients.map((recipient: AlertRecipient) => (
                <FormField
                  key={recipient.id}
                  control={form.control}
                  name={`${step}.${alertingType}` as Path<T>}
                  render={({ field }) => {
                    const label = getRecipientLabelText(recipient, alertingType);
                    const overflowThreshold = 25;
                    const isOverflowing = label.length > overflowThreshold;

                    return (
                      <FormItem className="hover:bg-muted flex flex-row items-center rounded-md border pl-3">
                        <FormControl>
                          <Checkbox
                            checked={field.value?.includes(recipient.id)}
                            onCheckedChange={(checked) => {
                              return checked
                                ? field.onChange([...(field.value || []), recipient.id])
                                : field.onChange(field.value?.filter((value: number) => value !== recipient.id));
                            }}
                          />
                        </FormControl>
                        {isOverflowing ? (
                          <Tooltip>
                            <TooltipTrigger asChild>
                              <FormLabel className="flex-1 cursor-pointer p-2 leading-5 font-normal">
                                {label.substring(0, overflowThreshold)}...
                              </FormLabel>
                            </TooltipTrigger>
                            <TooltipContent side="right" className="max-w-xs break-words">
                              {label}
                            </TooltipContent>
                          </Tooltip>
                        ) : (
                          <FormLabel className="flex-1 cursor-pointer p-2 leading-5 font-normal">{label}</FormLabel>
                        )}
                      </FormItem>
                    );
                  }}
                />
              ))}
            </div>
          ) : (
            <div className="mt-30 flex flex-col items-center justify-center">
              <p className="text-muted-foreground">
                {searchQuery.trim() ? t(`No ${recipientTypeText} found.`) : t(`No ${recipientTypeText} available.`)}
              </p>
            </div>
          )}
        </ScrollArea>
      </CardContent>
    </Card>
  );
}
