import {
  AlertDialog,
  AlertDialogTitle,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogTrigger,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogCancel,
} from '@/components/ui/alert-dialog';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm, UseFormReturn } from 'react-hook-form';
import { Input } from '@/components/ui/input';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Select, SelectItem, SelectContent, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useState } from 'react';
import { useUpdateCheck } from '@/components/alerting/queries';
import { AlertingType, FlowSourceCheckQuery, FlowSourceCheck } from '@/components/alerting/types';
import { Button } from '@/components/ui/button';
import { ClipboardCheckIcon, PencilIcon, XIcon } from 'lucide-react';
import { Source } from '@/components/source/types';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { DropdownMenuItem } from '@/components/ui/dropdown-menu';
import { useTranslation } from 'react-i18next';
import { type TFunction } from 'i18next';
import { SourceGroup } from '@/components/sourcegroups/types';
import CheckReceiverSelector from '../../CheckReceiverSelector';
import { flowSourceCheckFormSchema } from '@/components/alerting/schemas';

const getAssociationType = (check: FlowSourceCheck) => {
  if (check.object_type === 'source') {
    return 'source';
  } else if (check.object_type === 'sourcegroup') {
    return 'sourcegroup';
  }
};

const getAssociationId = (check: FlowSourceCheck, sources: Source[], sourcegroups: SourceGroup[]) => {
  if (check.object_type === 'source') {
    return sources.find((source) => source.id === check.object_id)?.id?.toString() || '';
  } else if (check.object_type === 'sourcegroup') {
    return sourcegroups.find((sourcegroup) => sourcegroup.id === check.object_id)?.id?.toString() || '';
  }
};

export default function EditFlowSourceCheckModal({
  sources,
  sourcegroups,
  check,
  onModalClose,
}: {
  sources: Source[];
  sourcegroups: SourceGroup[];
  check: FlowSourceCheck;
  onModalClose: () => void;
}) {
  const { t } = useTranslation();
  const { mutate: updateCheck } = useUpdateCheck();

  const form = useForm<z.infer<typeof flowSourceCheckFormSchema>>({
    resolver: zodResolver(flowSourceCheckFormSchema),
    mode: 'onTouched',
    defaultValues: {
      check_type: 'flow_source',
      step1: {
        name: check.name || '',
        association: {
          type: getAssociationType(check),
          id: getAssociationId(check, sources, sourcegroups),
        },
      },
      step2: {
        metric: check.metric || 'bytes',
        warning_threshold: check.warning?.toString() || '',
        critical_threshold: check.critical?.toString() || '',
        queries: parseRawQuery(check.raw_query),
      },
      step3: {
        user:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'user')
            .map((association) => association.association_id) || [],
        nagios:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'nagios')
            .map((association) => association.association_id) || [],
        snmp_receiver:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'snmp_receiver')
            .map((association) => association.association_id) || [],
        command:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'command')
            .map((association) => association.association_id) || [],
      },
    },
  });

  const [editCheckFormOpen, setEditCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2' | 'step3'>('step1');

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setEditCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
      form.trigger();
    } else {
      onModalClose();
      form.reset();
    }
  };

  // handle movement between each step
  const handleStepping = async ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2', 'step3'];
    const currentStepIndex = steps.indexOf(formStep);

    if (direction === 'next') {
      const isValid = await form.trigger(formStep);
      if (!isValid) return;
    }

    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2' | 'step3');
    }
  };

  // handle submit
  const onSubmit = (data: z.infer<typeof flowSourceCheckFormSchema>) => {
    updateCheck({
      id: check.id,
      data,
    });
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setEditCheckFormOpen(false);
    setFormStep('step1');
    form.reset();
  };

  return (
    <AlertDialog open={editCheckFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className="flex max-h-[800px] min-w-[800px] flex-col gap-6 overflow-hidden">
        <AlertDialogHeader className="flex h-min flex-col gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>
        <Form {...form}>
          {sources && (
            <form onSubmit={form.handleSubmit(onSubmit)} className="flex min-h-0 grow flex-col gap-4">
              {formStep === 'step1' && <Step1 form={form} sources={sources} sourcegroups={sourcegroups} t={t} />}
              {formStep === 'step2' && <Step2 form={form} t={t} />}
              {formStep === 'step3' && <Step3 form={form} t={t} />}
            </form>
          )}
        </Form>
        <AlertDialogFooter>
          <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
          {formStep !== 'step1' && (
            <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
              {t(`Previous`)}
            </Button>
          )}
          {formStep === 'step1' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step2' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step3' && (
            <Button className="w-20" onClick={() => onSubmit(form.getValues())}>
              {t(`Update`)}
            </Button>
          )}
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">{t(`Edit Flow Source Check`)}</span>
      <span className="ml-auto text-base font-medium">
        {formStep === 'step1' ? (
          <>{t('Step 1 - Select Flow Source')}</>
        ) : formStep === 'step2' ? (
          <>{t('Step 2 - Select Criteria')}</>
        ) : (
          formStep === 'step3' && <>{t('Step 3 - Select Alerting')}</>
        )}
      </span>
    </AlertDialogTitle>
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {formStep === 'step1' ? (
        <>{t('Select the flow source for the check. You can select a flow source group, or a single flow source.')}</>
      ) : formStep === 'step2' ? (
        <>
          {t(
            'Select the criteria you would like to check for. You can configure things like a single port, IP, or network.'
          )}
        </>
      ) : (
        formStep === 'step3' && (
          <>
            {t(
              'Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.'
            )}
          </>
        )
      )}
    </AlertDialogDescription>
  );
};

const Step1 = ({
  form,
  sources,
  sourcegroups,
  t,
}: {
  form: UseFormReturn<z.infer<typeof flowSourceCheckFormSchema>>;
  sources: Source[];
  sourcegroups: SourceGroup[];
  t: TFunction;
}) => {
  const [associationType, setAssociationType] = useState<'source' | 'sourcegroup'>(
    form.watch('step1.association.type') || 'source'
  );

  return (
    <div className="flex grow flex-col gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem>
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input
                {...field}
                onChange={(e) => {
                  form.setValue('step1.name', e.target.value, { shouldValidate: true });
                }}
              />
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <div className="flex flex-col gap-0">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="mb-0 gap-0">
              <FormLabel>{t('Check Association')}</FormLabel>
              <FormControl>
                <RadioGroup
                  defaultValue={field.value}
                  className="flex items-center gap-8 py-4"
                  onValueChange={(value) => {
                    field.onChange(value);
                    setAssociationType(value as 'source' | 'sourcegroup');
                  }}
                >
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="source" id="source" />
                    <Label htmlFor="source" className="cursor-pointer">
                      {t('Flow Source')}
                    </Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="sourcegroup" id="sourcegroup" />
                    <Label htmlFor="sourcegroup" className="cursor-pointer">
                      {t('Flow Source Group')}
                    </Label>
                  </div>
                </RadioGroup>
              </FormControl>
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="step1.association.id"
          render={({ field }) => (
            <FormItem>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => {
                    form.setValue('step1.association.id', value, { shouldValidate: true });
                    form.trigger('step1.association.type');
                  }}
                >
                  <SelectTrigger>
                    <SelectValue
                      placeholder={t(`Select a ${associationType === 'source' ? 'Source' : 'Source Group'}`)}
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {associationType === 'source'
                      ? sources.map((source) => (
                          <SelectItem key={source.id} value={source.id?.toString() ?? ''}>
                            {source.name}
                          </SelectItem>
                        ))
                      : sourcegroups.map((sourcegroup) => (
                          <SelectItem key={sourcegroup.id} value={sourcegroup.id?.toString() ?? ''}>
                            {sourcegroup.name}
                          </SelectItem>
                        ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof flowSourceCheckFormSchema>>; t: TFunction }) => {
  return (
    <div className="flex min-h-0 grow flex-col gap-4">
      <FormField
        control={form.control}
        name="step2.metric"
        render={({ field }) => (
          <FormItem className="flex flex-col px-2">
            <FormLabel>{t('Metric')}</FormLabel>
            <FormControl>
              <Select
                defaultValue={field.value}
                onValueChange={(value) => {
                  field.onChange(value);
                  form.trigger('step2.queries');
                }}
              >
                <SelectTrigger>
                  <SelectValue placeholder={t('Select a metric')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="bytes">{t('Bytes')}</SelectItem>
                  <SelectItem value="flows">{t('Flows')}</SelectItem>
                  <SelectItem value="packets">{t('Packets')}</SelectItem>
                  <SelectItem value="pps">{t('Packets/Sec')}</SelectItem>
                  <SelectItem value="bps">{t('Bits/Sec')}</SelectItem>
                  <SelectItem value="bpp">{t('Bytes/Packet')}</SelectItem>
                  {/* abnormal behavior unavailable until its confirmed working */}
                  {/* <SelectItem value="abnormal_behavior">{t('Abnormal Behavior')}</SelectItem> */}
                </SelectContent>
              </Select>
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <div className="flex flex-row gap-4 px-2">
        <FormField
          control={form.control}
          name="step2.warning_threshold"
          render={({ field }) => (
            <FormItem className="w-1/2">
              <FormLabel>{t('Warning Threshold')}</FormLabel>
              <FormControl>
                <Input
                  {...field}
                  className="disable-artifacts"
                  onChange={(e) => {
                    form.setValue('step2.warning_threshold', e.target.value.toString(), { shouldValidate: true });
                  }}
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.critical_threshold"
          render={({ field }) => (
            <FormItem className="w-1/2">
              <FormLabel>{t('Critical Threshold')}</FormLabel>
              <FormControl>
                <Input
                  {...field}
                  className="disable-artifacts"
                  onChange={(e) => {
                    form.setValue('step2.critical_threshold', e.target.value.toString(), { shouldValidate: true });
                  }}
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>

      {form.watch('step2.metric') !== 'abnormal_behavior' && (
        <div className="flex min-h-0 grow flex-col gap-2 overflow-y-auto">
          <p className="px-2.5 font-medium">{t('Queries')}</p>
          <div className="custom-scrollbar-gutters flex min-h-0 grow flex-col items-center gap-4 overflow-y-auto">
            <div className="flex w-full flex-col items-center gap-4">
              {form.watch('step2.queries').map((_, index) => (
                <CheckQueryBuilder key={index} form={form} index={index} t={t} />
              ))}
            </div>
            <Button
              type="button"
              variant="outline"
              className="w-min px-3 py-1.5"
              onClick={() => {
                form.setValue(
                  'step2.queries',
                  [
                    ...form.watch('step2.queries'),
                    {
                      location: 'destination',
                      location_type: 'port',
                      location_bool: 'is',
                      location_value: '',
                    },
                  ],
                  { shouldValidate: true }
                );
              }}
            >
              {t('and')}
            </Button>
          </div>
        </div>
      )}
    </div>
  );
};

const Step3 = ({ form, t }: { form: UseFormReturn<z.infer<typeof flowSourceCheckFormSchema>>; t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex grow flex-col gap-4">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {alertingTypes.map((type) => (
            <TabsTrigger key={type} value={type}>
              {t(
                type !== 'snmp_receiver'
                  ? type
                      .split('_')
                      .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                      .join(' ')
                  : 'SNMP Receiver'
              )}
            </TabsTrigger>
          ))}
        </TabsList>

        {alertingTypes.map((type) => (
          <TabsContent key={type} value={type}>
            <CheckReceiverSelector alertingType={type} form={form} step="step3" />
          </TabsContent>
        ))}
      </Tabs>
    </div>
  );
};

const CheckQueryBuilder = ({
  form,
  index,
  t,
}: {
  form: UseFormReturn<z.infer<typeof flowSourceCheckFormSchema>>;
  index: number;
  t: TFunction;
}) => {
  const getPlaceholder = (locationType: string) => {
    if (locationType === 'port') return '80';
    if (locationType === 'ip') return '192.168.1.1';
    if (locationType === 'network') return '192.168.1.0/24';
    return '';
  };

  const locationType = form.watch(`step2.queries.${index}.location_type`);
  const placeholder = getPlaceholder(locationType);

  return (
    <div className="bg-card flex w-full gap-2 rounded-md border p-4 duration-300">
      <FormField
        control={form.control}
        name={`step2.queries.${index}.location`}
        render={({ field }) => (
          <FormItem className="w-full">
            <FormControl>
              <Select
                defaultValue={field.value}
                onValueChange={(value) => field.onChange(value as 'destination' | 'source' | 'destination_or_source')}
              >
                <SelectTrigger>
                  <SelectValue placeholder={t('Select a location')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="destination">{t('Destination')}</SelectItem>
                  <SelectItem value="source">{t('Source')}</SelectItem>
                  <SelectItem value="destination_or_source">{t('Destination or Source')}</SelectItem>
                </SelectContent>
              </Select>
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />
      <FormField
        control={form.control}
        name={`step2.queries.${index}.location_type`}
        render={({ field }) => (
          <FormItem className="w-full">
            <FormControl>
              <Select
                defaultValue={field.value}
                onValueChange={(value) => {
                  field.onChange(value as 'port' | 'ip' | 'network');
                  form.trigger(`step2.queries.${index}.location_value`);
                }}
              >
                <SelectTrigger>
                  <SelectValue placeholder={t('Select a location type')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="port">{t('Port')}</SelectItem>
                  <SelectItem value="ip">{t('IP')}</SelectItem>
                  <SelectItem value="network">{t('Network')}</SelectItem>
                </SelectContent>
              </Select>
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <FormField
        control={form.control}
        name={`step2.queries.${index}.location_bool`}
        render={({ field }) => (
          <FormItem className="flex">
            <FormControl>
              <RadioGroup
                className="flex h-9 w-min gap-2 text-nowrap"
                defaultValue={field.value}
                onValueChange={(value) => field.onChange(value as 'is' | 'is_not')}
              >
                <div className="flex items-center gap-2">
                  <RadioGroupItem value="is" id="is" />
                  <Label htmlFor="is" className="cursor-pointer">
                    {t('is')}
                  </Label>
                </div>
                <div className="flex items-center gap-2">
                  <RadioGroupItem value="is_not" id="is_not" />
                  <Label htmlFor="is_not" className="cursor-pointer">
                    {t('is not')}
                  </Label>
                </div>
              </RadioGroup>
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <FormField
        control={form.control}
        name={`step2.queries.${index}.location_value`}
        render={({ field }) => (
          <FormItem className="flex w-full flex-col">
            <FormControl>
              <Input
                placeholder={placeholder}
                {...field}
                onChange={(e) => {
                  form.setValue(`step2.queries.${index}.location_value`, e.target.value, { shouldValidate: true });
                }}
              />
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      {form.watch('step2.queries').length > 1 && (
        <Button
          type="button"
          variant="ghost"
          size="icon"
          className="cursor-pointer px-6 py-2"
          onClick={() => {
            form.setValue(
              'step2.queries',
              form.watch('step2.queries').filter((_, i) => i !== index)
            );
            form.trigger('step2.queries');
          }}
        >
          <XIcon />
        </Button>
      )}
    </div>
  );
};

const parseRawQuery = (raw_query: string | undefined): FlowSourceCheckQuery[] => {
  // default fallback
  if (!raw_query || raw_query.trim() === '') {
    return [
      {
        location: 'destination',
        location_type: 'port',
        location_bool: 'is',
        location_value: '',
      },
    ];
  }

  // split by " and " to get individual query parts
  const query_parts = raw_query.split(' and ');
  const queries_array: FlowSourceCheckQuery[] = [];

  query_parts.forEach((query_part) => {
    const trimmed_query = query_part.trim();

    // determine if it's a "not" query
    const is_not_query = trimmed_query.startsWith('not ');
    const query_without_not = is_not_query ? trimmed_query.substring(4) : trimmed_query;

    // parse location (src, dst, or empty for destination_or_source)
    let location: 'source' | 'destination' | 'destination_or_source' = 'destination';
    let remaining_query = query_without_not;

    if (query_without_not.startsWith('src ')) {
      location = 'source';
      remaining_query = query_without_not.substring(4);
    } else if (query_without_not.startsWith('dst ')) {
      location = 'destination';
      remaining_query = query_without_not.substring(4);
    } else {
      // if no src/dst prefix, it's destination_or_source
      location = 'destination_or_source';
    }

    // parse location_type and value
    let location_type: 'port' | 'ip' | 'network' = 'port';
    let location_value = '';

    if (remaining_query.startsWith('port ')) {
      location_type = 'port';
      location_value = remaining_query.substring(5).trim();
    } else if (remaining_query.startsWith('ip ')) {
      location_type = 'ip';
      location_value = remaining_query.substring(3).trim();
    } else if (remaining_query.startsWith('net ')) {
      location_type = 'network';
      location_value = remaining_query.substring(4).trim();
    } else {
      // fallback to port if no type specified
      location_type = 'port';
      location_value = remaining_query.trim();
    }

    queries_array.push({
      location,
      location_type,
      location_bool: is_not_query ? 'is_not' : 'is',
      location_value,
    });
  });

  return queries_array;
};
