import { Form, FormField, FormLabel, FormItem, FormControl, FormMessage } from '@/components/ui/form';
import { ClipboardCheckIcon, PencilIcon } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm, UseFormReturn } from 'react-hook-form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import {
  AlertDialog,
  AlertDialogTitle,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogTrigger,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogCancel,
} from '@/components/ui/alert-dialog';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useGetScheduledScans } from '@/components/nmap/scheduledscans/queries';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { type scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { TFunction } from 'i18next';
import { useUpdateCheck } from '@/components/alerting/queries';
import { AlertingType, NmapCheck } from '../../types';
import { DropdownMenuItem } from '@/components/ui/dropdown-menu';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import CheckReceiverSelector from '../../CheckReceiverSelector';
import { nmapCheckFormSchema } from '@/components/alerting/schemas';

export default function EditNmapCheckModal({ check, onModalClose }: { check: NmapCheck; onModalClose: () => void }) {
  const { t } = useTranslation();
  const { data: nmapScans } = useGetScheduledScans();
  const { mutate: updateCheck } = useUpdateCheck();

  const [editNmapCheckFormOpen, setEditNmapCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2' | 'step3'>('step1');

  const form = useForm<z.infer<typeof nmapCheckFormSchema>>({
    resolver: zodResolver(nmapCheckFormSchema),
    mode: 'onTouched',
    defaultValues: {
      check_type: 'nmap',
      step1: {
        name: check.name,
        association: {
          type: check.object_type as 'scheduled_scan',
          id: check.object_id.toString(),
        },
      },
      step2: {
        metric: check.metric || 'ports_open',
        warning_threshold: check.warning ?? '',
        critical_threshold: check.critical ?? '',
      },
      step3: {
        user:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'user')
            .map((association) => association.association_id) || [],
        nagios:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'nagios')
            .map((association) => association.association_id) || [],
        snmp_receiver:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'snmp_receiver')
            .map((association) => association.association_id) || [],
        command:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'command')
            .map((association) => association.association_id) || [],
      },
    },
  });

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setEditNmapCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
      form.trigger();
    } else {
      onModalClose();
      form.reset();
    }
  };

  // handle movement between each step
  const handleStepping = async ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2', 'step3'];
    const currentStepIndex = steps.indexOf(formStep);

    if (direction === 'next') {
      const isValid = await form.trigger(formStep);
      if (!isValid) return;
    }

    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2' | 'step3');
    }
  };

  // handle submit
  const handleSubmit = (data: z.infer<typeof nmapCheckFormSchema>) => {
    updateCheck({ id: check.id, data });
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setEditNmapCheckFormOpen(false);
    setFormStep('step1');
    form.reset();
  };

  return (
    <AlertDialog open={editNmapCheckFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className="flex max-h-[800px] min-w-[800px] flex-col gap-6 overflow-hidden">
        <AlertDialogHeader className="flex h-min flex-col gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex min-h-0 grow flex-col gap-4">
            {formStep === 'step1' && <Step1 form={form} nmapScans={nmapScans ?? []} t={t} />}
            {formStep === 'step2' && <Step2 form={form} t={t} />}
            {formStep === 'step3' && <Step3 form={form} t={t} />}
          </form>
        </Form>
        <AlertDialogFooter>
          <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
          {formStep !== 'step1' && (
            <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
              {t(`Previous`)}
            </Button>
          )}
          {formStep === 'step1' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step2' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step3' && (
            <Button
              className="w-20"
              onClick={() => handleSubmit(form.getValues())}
              disabled={form.formState.isSubmitting}
            >
              {t(`Update`)}
            </Button>
          )}
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">{t(`Edit Nmap Check`)}</span>
      <span className="ml-auto text-base font-medium">
        {formStep === 'step1' ? (
          <>{t(`Step 1 - Select Scheduled Scan`)}</>
        ) : formStep === 'step2' ? (
          <>{t(`Step 2 - Select Criteria`)}</>
        ) : (
          formStep === 'step3' && <>{t(`Step 3 - Select Alerting`)}</>
        )}
      </span>
    </AlertDialogTitle>
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {formStep === 'step1' ? (
        <>{t(`Select the scheduled scan for the check.`)}</>
      ) : formStep === 'step2' ? (
        <>{t(`Select the criteria you would like to check for.`)}</>
      ) : (
        formStep === 'step3' && (
          <>
            {t(
              `Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.`
            )}
          </>
        )
      )}
    </AlertDialogDescription>
  );
};

const Step1 = ({
  form,
  nmapScans,
  t,
}: {
  form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>;
  nmapScans: scheduledNmapScan[];
  t: TFunction;
}) => {
  const [associationType, setAssociationType] = useState<'scheduled_scan'>('scheduled_scan');

  return (
    <div className="flex grow flex-col gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem>
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input {...field} />
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <div className="flex flex-col gap-0">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="mb-0 gap-0">
              <FormLabel>{t('Check Association')}</FormLabel>
              <FormControl>
                <RadioGroup
                  defaultValue="scheduled_scan"
                  className="flex items-center gap-8 py-4"
                  onValueChange={(value) => {
                    field.onChange(value);
                    setAssociationType(value as 'scheduled_scan');
                  }}
                >
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="scheduled_scan" id="scheduled_scan" />
                    <Label htmlFor="scheduled_scan" className="cursor-pointer">
                      {t('Scheduled Scan')}
                    </Label>
                  </div>
                </RadioGroup>
              </FormControl>
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="step1.association.id"
          render={({ field }) => (
            <FormItem>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => {
                    form.setValue('step1.association.id', value, { shouldValidate: true });
                    form.trigger('step1.association.type');
                  }}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t(`Select a Scheduled Scan`)} />
                  </SelectTrigger>
                  <SelectContent>
                    {associationType === 'scheduled_scan' &&
                      nmapScans?.map((scan) => (
                        <SelectItem key={scan.id} value={scan.id?.toString() ?? ''}>
                          {scan.name}
                        </SelectItem>
                      ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>; t: TFunction }) => {
  return (
    <div className="flex grow flex-row gap-8">
      <FormField
        control={form.control}
        name="step2.metric"
        render={({ field }) => (
          <FormItem className="w-full">
            <FormLabel>{t('Metric')}</FormLabel>
            <FormControl>
              <Select defaultValue={field.value} onValueChange={(value) => field.onChange(value)}>
                <SelectTrigger>
                  <SelectValue placeholder={t('Select a metric')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ports_open">{t('Ports Open')}</SelectItem>
                  <SelectItem value="ports_closed">{t('Ports Closed')}</SelectItem>
                </SelectContent>
              </Select>
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />
      <div className="flex w-full flex-row gap-4">
        <FormField
          control={form.control}
          name="step2.warning_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Warning Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.critical_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Critical Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step3 = ({ form, t }: { form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>; t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex grow flex-col gap-4">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {alertingTypes.map((type) => (
            <TabsTrigger key={type} value={type}>
              {t(
                type !== 'snmp_receiver'
                  ? type
                      .split('_')
                      .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                      .join(' ')
                  : 'SNMP Receiver'
              )}
            </TabsTrigger>
          ))}
        </TabsList>

        {alertingTypes.map((type) => (
          <TabsContent key={type} value={type}>
            <CheckReceiverSelector alertingType={type} form={form} step="step3" />
          </TabsContent>
        ))}
      </Tabs>
    </div>
  );
};
