import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { Button } from '@/components/ui/button';
import { HardDriveIcon, PlusIcon } from 'lucide-react';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { useState } from 'react';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Checkbox } from '@/components/ui/checkbox';
import { useCreateNagiosServer } from '@/components/alerting/queries';
import { NagiosServerFormData } from '@/components/alerting/types';
import { useTranslation } from 'react-i18next';
import { useAuth } from '@/contexts/AuthContext';
import { isAdmin } from '@/components/role/utils';

const nagiosServerFormSchema = z.object({
  name: z.string().min(1),
  nrdp_url: z.string().url(),
  nrdp_token: z.string().min(1),
  verify_ssl: z.boolean().default(true),
});

export default function AddNagiosServerModal() {
  const { t } = useTranslation();
  const [newNagiosServerFormOpen, setNewNagiosServerFormOpen] = useState(false);
  const { mutate: createNagiosServer } = useCreateNagiosServer();
  const { user } = useAuth();

  const form = useForm({
    resolver: zodResolver(nagiosServerFormSchema),
    mode: 'onChange',
    defaultValues: {
      name: '',
      nrdp_url: '',
      nrdp_token: '',
      verify_ssl: true,
    },
  });

  const handleSubmit = (data: NagiosServerFormData) => {
    createNagiosServer(data);
  };

  const handleFormOpen = (open: boolean) => {
    setNewNagiosServerFormOpen(open);
    if (open) {
      form.reset();
    }
  };

  const handleCancel = () => {
    handleFormOpen(false);
    form.reset();
  };

  return (
    <AlertDialog open={newNagiosServerFormOpen} onOpenChange={handleFormOpen}>
      {isAdmin(user?.role) && (
        <AlertDialogTrigger asChild>
          <Button className="w-fit text-xs font-bold">
            <PlusIcon />
            {t(`New Nagios Server`)}
          </Button>
        </AlertDialogTrigger>
      )}
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
            <HardDriveIcon size={20} />
            <span className="text-lg font-medium">{t(`Add Nagios Server`)}</span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Add a new Nagios server to NNA, allowing you to send alerts directly to your Nagios server.`)}
        </AlertDialogDescription>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex min-h-0 grow flex-col gap-4">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_url"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP URL`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_token"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP Token`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="verify_ssl"
              render={({ field }) => (
                <FormItem className="flex flex-row items-start space-y-0 space-x-3">
                  <FormControl>
                    <Checkbox checked={field.value} onCheckedChange={field.onChange} />
                  </FormControl>
                  <div className="space-y-1 leading-none">
                    <FormLabel>{t(`Verify SSL Certificate`)}</FormLabel>
                    <FormDescription>
                      {t(
                        `Disable this only for self-signed certificates or trusted internal servers if NRDP is serving over a secure connection.`
                      )}
                    </FormDescription>
                  </div>
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <AlertDialogAction
                disabled={!form.formState.isValid || form.formState.isSubmitting}
                type="submit"
                className="w-fit"
              >
                {t(`Add Nagios Server`)}
              </AlertDialogAction>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  );
}
