import { PencilIcon, SquareTerminalIcon } from 'lucide-react';
import { useState } from 'react';
import {
  AlertDialog,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { z } from 'zod';
import { useForm } from 'react-hook-form';
import { FormField, FormItem, FormLabel, FormControl, FormMessage, Form } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { zodResolver } from '@hookform/resolvers/zod';
import { Command, CommandFormData } from '@/components/alerting/types';
import { useUpdateCommand } from '@/components/alerting/queries';
import { DropdownMenuItem } from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { useTranslation } from 'react-i18next';

const commandFormSchema = z.object({
  name: z.string().min(1, 'Command name required').max(255, 'Must be less than 255 characters'),
  location: z.string().min(1, 'Command required'),
  script: z.string(),
  arguments: z.string().optional(),
});

export default function EditCommandModal({ command, onModalClose }: { command: Command; onModalClose: () => void }) {
  const { t } = useTranslation();
  const { mutate: updateCommand } = useUpdateCommand();
  const [editCommandFormOpen, setEditCommandFormOpen] = useState(false);

  const form = useForm({
    resolver: zodResolver(commandFormSchema),
    mode: 'onBlur',
    defaultValues: {
      name: command.name || '',
      location: command.location || '',
      script: command.script || '',
      arguments: command.arguments || '',
    },
  });

  const onSubmit = (data: CommandFormData) => {
    updateCommand({ id: command.id, data });
    handleFormOpen(false);
  };

  const handleFormOpen = (open: boolean) => {
    setEditCommandFormOpen(open);
    if (open) {
      form.reset();
    }
    if (!open) {
      onModalClose();
    }
  };

  const handleCancel = () => {
    handleFormOpen(false);
    form.reset();
  };

  return (
    <AlertDialog open={editCommandFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className="flex min-w-[800px] flex-col">
        <AlertDialogHeader>
          <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
            <SquareTerminalIcon size={20} />
            <span className="text-lg font-medium">{t(`Edit Command`)}</span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>{t(`Edit an existing command to be used in your checks.`)}</AlertDialogDescription>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col gap-4">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Command Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder={t('Enter command name')} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="location"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Script Location`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder={t('Script location (e.g. /usr/bin)')} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="script"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Script Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder={t('Script name (e.g. notify.sh)')} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="arguments"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Passed Arguments`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder={t('Optional arguments')} />
                  </FormControl>
                  <FormMessage />
                  <div className="text-muted-foreground flex flex-col gap-2">
                    {t(
                      `You can pass some basic macros to the script via arguments that will be auto-populated when the script is executed.`
                    )}
                    <ul className="list-inside list-disc">
                      <li>
                        {`%sourcename%`} - {t(`the name of the source that is being alerted on`)}
                      </li>
                      <li>
                        {`%state%`} - {t(`the state of the check`)}
                      </li>
                      <li>
                        {`%returncode%`} - {t(`the return code of the check`)}
                      </li>
                      <li>
                        {`%output%`} - {t(`the output of the check`)}
                      </li>
                    </ul>
                  </div>
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel type="button" onClick={handleCancel}>
                {t(`Cancel`)}
              </AlertDialogCancel>
              <Button type="submit" disabled={!form.formState.isValid || form.formState.isSubmitting} className="w-fit">
                {t(`Update Command`)}
              </Button>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  );
}
