import { Form, FormField, FormLabel, FormItem, FormControl, FormMessage } from '@/components/ui/form';
import { ClipboardCheckIcon, PencilIcon } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm, UseFormReturn } from 'react-hook-form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import {
  AlertDialog,
  AlertDialogTitle,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogTrigger,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogCancel,
} from '@/components/ui/alert-dialog';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { TFunction } from 'i18next';
import { useUpdateCheck } from '@/components/alerting/queries';
import { AlertingType, SuricataCheck } from '../../types';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import { DropdownMenuItem } from '@/components/ui/dropdown-menu';
import CheckReceiverSelector from '../../CheckReceiverSelector';
import { suricataCheckFormSchema } from '@/components/alerting/schemas';

export default function EditSuricataCheckModal({
  check,
  onModalClose,
}: {
  check: SuricataCheck;
  onModalClose: () => void;
}) {
  const { t } = useTranslation();
  const { mutate: updateCheck } = useUpdateCheck();

  const [newSuricataCheckFormOpen, setNewSuricataCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2'>('step1');

  const form = useForm<z.infer<typeof suricataCheckFormSchema>>({
    resolver: zodResolver(suricataCheckFormSchema),
    mode: 'onTouched',
    defaultValues: {
      check_type: check.check_type,
      step1: {
        name: check.name,
        association: {
          type: 'signature_id',
          id: check.object_id.toString(),
        },
        metric: check.metric,
        warning_threshold: check.warning ?? '',
        critical_threshold: check.critical ?? '',
        lookback_period: getSuricataCheckQueryParam(check.raw_query, 'lookback_period') || '1d',
        check_frequency: getSuricataCheckQueryParam(check.raw_query, 'frequency') || '5min',
      },
      step2: {
        user:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'user')
            .map((association) => association.association_id) || [],
        nagios:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'nagios')
            .map((association) => association.association_id) || [],
        snmp_receiver:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'snmp_receiver')
            .map((association) => association.association_id) || [],
        command:
          check.alerting_associations
            ?.filter((association) => association.association_type === 'command')
            .map((association) => association.association_id) || [],
      },
    },
  });

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setNewSuricataCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
      form.trigger();
    } else {
      onModalClose();
      form.reset();
    }
  };

  // handle movement between each step
  const handleStepping = async ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2'];
    const currentStepIndex = steps.indexOf(formStep);

    if (direction === 'next') {
      const isValid = await form.trigger(formStep);
      if (!isValid) return;
    }

    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2');
    }
  };

  // handle submit
  const handleSubmit = (data: z.infer<typeof suricataCheckFormSchema>) => {
    updateCheck({ id: check.id, data });
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setNewSuricataCheckFormOpen(false);
    setFormStep('step1');
    form.reset();
  };

  return (
    <AlertDialog open={newSuricataCheckFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className="flex max-h-[800px] min-w-[800px] flex-col gap-6 overflow-hidden">
        <AlertDialogHeader className="flex h-min flex-col gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex min-h-0 grow flex-col gap-4">
            {formStep === 'step1' && <Step1 form={form} t={t} />}
            {formStep === 'step2' && <Step2 form={form} t={t} />}
          </form>
        </Form>
        <AlertDialogFooter>
          <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
          {formStep !== 'step1' && (
            <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
              {t(`Previous`)}
            </Button>
          )}
          {formStep === 'step1' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step2' && (
            <Button
              className="w-20"
              onClick={() => handleSubmit(form.getValues())}
              disabled={form.formState.isSubmitting}
            >
              {t(`Update`)}
            </Button>
          )}
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2'; t: TFunction }) => {
  return (
    <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">{t(`Edit Suricata Check`)}</span>
      <span className="ml-auto text-base font-medium">
        {formStep === 'step1' ? <>{t(`Step 1 - Select Criteria`)}</> : <>{t(`Step 2 - Select Alerting`)}</>}
      </span>
    </AlertDialogTitle>
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2'; t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {formStep === 'step1' ? (
        <>{t(`Select the criteria you would like to check for.`)}</>
      ) : (
        <>
          {t(
            `Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.`
          )}
        </>
      )}
    </AlertDialogDescription>
  );
};

const Step1 = ({ form, t }: { form: UseFormReturn<z.infer<typeof suricataCheckFormSchema>>; t: TFunction }) => {
  return (
    <div className="flex grow flex-col gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem className="relative">
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input {...field} />
            </FormControl>
            <FormMessage className="absolute -bottom-6" />
          </FormItem>
        )}
      />

      <div className="flex w-full flex-row gap-8">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Check On')}</FormLabel>
              <FormControl>
                <Select defaultValue={field.value} onValueChange={(value) => field.onChange(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a check on')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="signature_id">{t('Signature ID')}</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step1.association.id"
          render={({ field }) => (
            <FormItem className="relative w-full">
              <FormLabel>{t('Signature ID')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage className="absolute -bottom-6" />
            </FormItem>
          )}
        />
      </div>

      <div className="flex grow flex-row gap-8">
        <FormField
          control={form.control}
          name="step1.metric"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Metric')}</FormLabel>
              <FormControl>
                <Select defaultValue={field.value} onValueChange={(value) => field.onChange(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a metric')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="alert_count">{t('Alert Count')}</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <div className="flex w-full flex-row gap-4">
          <FormField
            control={form.control}
            name="step1.warning_threshold"
            render={({ field }) => (
              <FormItem className="w-full">
                <FormLabel>{t('Warning Threshold')}</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="step1.critical_threshold"
            render={({ field }) => (
              <FormItem className="w-full">
                <FormLabel>{t('Critical Threshold')}</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>
      </div>

      <div className="flex grow flex-row gap-8">
        <FormField
          control={form.control}
          name="step1.check_frequency"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Check Frequency')}</FormLabel>
              <FormControl>
                <Select defaultValue={field.value} onValueChange={(value) => field.onChange(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a frequency')} />
                  </SelectTrigger>
                  <SelectContent>
                    {check_frequency_times.map((time) => (
                      <SelectItem key={time.value} value={time.value}>
                        {t(time.label)}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step1.lookback_period"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Lookback Period')}</FormLabel>
              <FormControl>
                <Select defaultValue={field.value} onValueChange={(value) => field.onChange(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a lookback period')} />
                  </SelectTrigger>
                  <SelectContent>
                    {lookback_period_times.map((time) => (
                      <SelectItem key={time.value} value={time.value}>
                        {t(time.label)}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof suricataCheckFormSchema>>; t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex grow flex-col gap-4">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {alertingTypes.map((type) => (
            <TabsTrigger key={type} value={type}>
              {t(
                type !== 'snmp_receiver'
                  ? type
                      .split('_')
                      .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                      .join(' ')
                  : 'SNMP Receiver'
              )}
            </TabsTrigger>
          ))}
        </TabsList>

        {alertingTypes.map((type) => (
          <TabsContent key={type} value={type}>
            <CheckReceiverSelector alertingType={type} form={form} step="step2" />
          </TabsContent>
        ))}
      </Tabs>
    </div>
  );
};

const check_frequency_times = [
  {
    value: '5min',
    label: 'Every 5 Minutes',
  },
  {
    value: 'Hourly',
    label: 'Hourly',
  },
  {
    value: 'Daily',
    label: 'Daily',
  },
  {
    value: 'Weekly',
    label: 'Weekly',
  },
  {
    value: 'Monthly',
    label: 'Monthly',
  },
  {
    value: 'Yearly',
    label: 'Yearly',
  },
];

const lookback_period_times = [
  {
    value: '1d',
    label: '1 Day',
  },
  {
    value: '7d',
    label: '7 Days',
  },
  {
    value: '30d',
    label: '30 Days',
  },
  {
    value: '90d',
    label: '90 Days',
  },
  {
    value: '180d',
    label: '180 Days',
  },
  {
    value: '365d',
    label: '365 Days',
  },
];

const getSuricataCheckQueryParam = (raw_query: string, param: string) => {
  const query_params = raw_query.split('&');
  const query_params_obj: { [key: string]: string } = {};
  query_params.forEach((param: string) => {
    const [key, value] = param.split('=');
    query_params_obj[key] = value;
  });
  return query_params_obj[param];
};
