import { Form, FormField, FormLabel, FormItem, FormControl, FormMessage } from '@/components/ui/form';
import { ClipboardCheckIcon, PlusIcon } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm, UseFormReturn } from 'react-hook-form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import {
  AlertDialog,
  AlertDialogTitle,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogTrigger,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogCancel,
} from '@/components/ui/alert-dialog';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { TFunction } from 'i18next';
import { useCreateCheck } from '@/components/alerting/queries';
import { AlertingType } from '@/components/alerting/types';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import CheckReceiverSelector from '@/components/alerting/CheckReceiverSelector';
import { useAuth } from '@/contexts/AuthContext';
import { isAdmin } from '@/components/role/utils';
import { Switch } from '@/components/ui/switch';
import { systemCheckFormSchema } from '@/components/alerting/schemas';

const resourceMetrics: Record<'cpu' | 'memory' | 'drive', string[]> = {
  cpu: ['cpu_usage'],
  memory: ['used_percent', 'swap_percent'],
  drive: ['percent', 'available'],
};

const metricLabels: Record<string, string> = {
  cpu_usage: 'CPU Usage (%)',
  used_percent: 'Memory Used (%)',
  swap_percent: 'Memory Swap (%)',
  available: 'Storage Available (bytes)',
  percent: 'Storage Used (%)',
};

export default function AddSystemCheckModal() {
  const { t } = useTranslation();
  const { user } = useAuth();
  const createCheck = useCreateCheck();
  const [newSystemCheckFormOpen, setNewSystemCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2' | 'step3'>('step1');

  const form = useForm({
    resolver: zodResolver(systemCheckFormSchema),
    mode: 'onTouched',
    defaultValues: {
      check_type: 'system',
      step1: {
        name: '',
        association: {
          type: 'cpu',
          id: '-1',
        },
        metric: 'cpu_usage',
      },
      step2: {
        forcecheck: false,
        warning_threshold: '',
        critical_threshold: '',
      },
      step3: {
        user: [],
        nagios: [],
        snmp_receiver: [],
        command: [],
      },
    },
  });

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setNewSystemCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
    } else {
      form.reset();
    }
  };

  // handle movement between each step
  const handleStepping = async ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2', 'step3'];
    const currentStepIndex = steps.indexOf(formStep);

    if (direction === 'next') {
      const isValid = await form.trigger(formStep);
      if (!isValid) return;
    }

    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2' | 'step3');
    }
  };

  // handle submit
  const handleSubmit = (data: z.infer<typeof systemCheckFormSchema>) => {
    createCheck.mutate(data);
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setNewSystemCheckFormOpen(false);
    setFormStep('step1');
    form.reset();
  };

  return (
    <AlertDialog open={newSystemCheckFormOpen} onOpenChange={handleFormOpen}>
      {isAdmin(user?.role) && (
        <AlertDialogTrigger asChild>
          <Button className="w-fit">
            <PlusIcon />
            {t('New System Check')}
          </Button>
        </AlertDialogTrigger>
      )}
      <AlertDialogContent className="flex max-h-[800px] min-w-[800px] flex-col gap-6 overflow-hidden">
        <AlertDialogHeader className="flex h-min flex-col gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex min-h-0 grow flex-col gap-4">
            {formStep === 'step1' && <Step1 form={form} t={t} />}
            {formStep === 'step2' && <Step2 form={form} t={t} />}
            {formStep === 'step3' && <Step3 form={form} t={t} />}
          </form>
        </Form>
        <AlertDialogFooter>
          <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
          {formStep !== 'step1' && (
            <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
              {t(`Previous`)}
            </Button>
          )}
          {formStep === 'step1' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step2' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step3' && (
            <Button
              className="w-20"
              onClick={() => handleSubmit(form.getValues())}
              disabled={form.formState.isSubmitting}
            >
              {t(`Create`)}
            </Button>
          )}
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogTitle className="mb-2 flex h-min w-full items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">{t(`Add System Check`)}</span>
      <span className="ml-auto text-base font-medium">
        {formStep === 'step1' ? (
          <>{t(`Step 1 - Select Resource and Metric`)}</>
        ) : formStep === 'step2' ? (
          <>{t(`Step 2 - Select Thresholds`)}</>
        ) : (
          formStep === 'step3' && <>{t(`Step 3 - Select Alerting`)}</>
        )}
      </span>
    </AlertDialogTitle>
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3'; t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {formStep === 'step1' ? (
        <>{t(`Select the resource for the check.`)}</>
      ) : formStep === 'step2' ? (
        <>{t(`Select the thresholds for the check.`)}</>
      ) : (
        formStep === 'step3' && (
          <>
            {t(
              `Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.`
            )}
          </>
        )
      )}
    </AlertDialogDescription>
  );
};

const Step1 = ({ form, t }: { form: UseFormReturn<z.infer<typeof systemCheckFormSchema>>; t: TFunction }) => {
  const [associationType, setAssociationType] = useState<'cpu' | 'memory' | 'drive'>(
    form.watch('step1.association.type') || 'cpu'
  );

  const defaultMetricByType: Record<string, string> = {
    cpu: 'cpu_usage',
    memory: 'used_percent',
    drive: 'percent',
  };

  return (
    <div className="flex grow flex-col gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem>
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input {...field} />
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <div className="flex flex-col gap-0">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="mb-0 gap-0">
              <FormLabel>{t('Check Association')}</FormLabel>
              <FormControl>
                <RadioGroup
                  value={associationType}
                  className="flex items-center gap-8 py-4"
                  onValueChange={(value) => {
                    const newType = value as 'cpu' | 'memory' | 'drive';
                    field.onChange(newType);
                    setAssociationType(newType);
                    form.setValue('step1.metric', defaultMetricByType[newType], { shouldValidate: true });
                  }}
                >
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="cpu" id="cpu" />
                    <Label htmlFor="cpu" className="cursor-pointer">
                      {t('CPU')}
                    </Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="memory" id="memory" />
                    <Label htmlFor="memory" className="cursor-pointer">
                      {t('Memory')}
                    </Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="drive" id="drive" />
                    <Label htmlFor="drive" className="cursor-pointer">
                      {t('Root Drive')}
                    </Label>
                  </div>
                </RadioGroup>
              </FormControl>
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="step1.metric"
          render={({ field }) => (
            <FormItem>
              <FormControl>
                <Select key={associationType} value={field.value} onValueChange={(value) => field.onChange(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder={t(`Select a Metric`)} />
                  </SelectTrigger>
                  <SelectContent>
                    {resourceMetrics[associationType].map((metric) => (
                      <SelectItem value={metric} key={metric}>
                        {t(metricLabels[metric])}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof systemCheckFormSchema>>; t: TFunction }) => {
  return (
    <div className="flex grow flex-row gap-8">
      <div className="flex w-full flex-row gap-4">
        <FormField
          control={form.control}
          name="step2.warning_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Warning Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.critical_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Critical Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.forcecheck"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormControl>
                <div className="mt-5 flex flex-row items-center gap-2">
                  <Switch checked={!!field.value} onCheckedChange={field.onChange} />
                  <FormLabel className="m-0 cursor-pointer select-none">{t('Force Check on Creation')}</FormLabel>
                </div>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  );
};

const Step3 = ({ form, t }: { form: UseFormReturn<z.infer<typeof systemCheckFormSchema>>; t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex grow flex-col gap-4">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {alertingTypes.map((type) => (
            <TabsTrigger key={type} value={type}>
              {t(
                type !== 'snmp_receiver'
                  ? type
                      .split('_')
                      .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                      .join(' ')
                  : 'SNMP Receiver'
              )}
            </TabsTrigger>
          ))}
        </TabsList>

        {alertingTypes.map((type) => (
          <TabsContent key={type} value={type}>
            <CheckReceiverSelector alertingType={type} form={form} step="step3" />
          </TabsContent>
        ))}
      </Tabs>
    </div>
  );
};
