import { useMutation, useQuery } from '@tanstack/react-query';
import { useQueryClient } from '@tanstack/react-query';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import axios, { AxiosError } from '@/lib/axios';
import {
  Command,
  CommandFormData,
  FlowSourceCheckFormData,
  NagiosServer,
  NagiosServerFormData,
  NmapCheckFormData,
  ServiceHostname,
  ServiceHostnameFormData,
  SNMPReceiver,
  SNMPReceiverFormData,
  SuricataCheckFormData,
  SystemCheckFormData,
} from '@/components/alerting/types';

// check queries
export function useCreateCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: FlowSourceCheckFormData | NmapCheckFormData | SuricataCheckFormData | SystemCheckFormData) => {
      return axios.post('/api/v1/checks', data);
    },
    onSuccess: (_, variables) => {
      queryClient.invalidateQueries({ queryKey: ['checks', variables.check_type] });
      toast(t('Check successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create check.'), { description: t('Please try again.') });
    },
  });
}

export function useUpdateCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({
      id,
      data,
    }: {
      id: number;
      data: FlowSourceCheckFormData | NmapCheckFormData | SuricataCheckFormData | SystemCheckFormData;
    }) => {
      return axios.put(`/api/v1/checks/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Check successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update check.'), { description: t('Please try again.') });
    },
  });
}

export function useGetChecks({ check_type }: { check_type: 'flow_source' | 'nmap' | 'suricata' | 'system' }) {
  return useQuery({
    queryKey: ['checks', check_type],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/checks?check_type=${check_type}`);
      return response.data;
    },
  });
}

export function useDeleteCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/checks/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Check successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete check.'), { description: t('Please try again.') });
    },
  });
}

export function useForceCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.post(`/api/v1/checks/${id}/force`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Running check now!'));
    },
    onError: () => {
      toast(t('Failed to force check.'), { description: t('Please try again.') });
    },
  });
}

// TODO: ADD OPTIMISTIC UPDATES TO THIS
export function useToggleActiveCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.patch(`/api/v1/checks/${id}/toggle`);
    },
    onSuccess: (res) => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t(`Check successfully ${res.data.active ? 'Activated' : 'Deactivated'}.`));
    },
    onError: () => {
      toast(t(`Failed to update check active status.`), { description: t('Please try again.') });
    },
  });
}

// command queries
export function useCreateCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: CommandFormData) => {
      return axios.post('/api/v1/commands', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create command.'), { description: t('Please try again.') });
    },
  });
}

export function useGetCommands() {
  return useQuery<Command[], AxiosError>({
    queryKey: ['commands'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/commands');
      return response.data;
    },
  });
}

export function useUpdateCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number; data: CommandFormData }) => {
      return axios.put(`/api/v1/commands/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update command.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/commands/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete command.'), { description: t('Please try again.') });
    },
  });
}

// nagios server queries
export function useCreateNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: NagiosServerFormData) => {
      return axios.post('/api/v1/nagios-servers', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create Nagios server.'), { description: t('Please try again.') });
    },
  });
}

export function useGetNagiosServers() {
  return useQuery<NagiosServer[], AxiosError>({
    queryKey: ['nagios-servers'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/nagios-servers');
      return response.data;
    },
  });
}

export function useUpdateNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number; data: NagiosServerFormData }) => {
      return axios.put(`/api/v1/nagios-servers/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update Nagios server.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/nagios-servers/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete Nagios server.'), { description: t('Please try again.') });
    },
  });
}

// nagios server service/hostnames queries
export function useCreateServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: ServiceHostnameFormData) => {
      return axios.post('/api/v1/service-hostnames', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create Service/Hostname.'), { description: t('Please try again.') });
    },
  });
}

export function useGetServiceHostnames() {
  return useQuery<ServiceHostname[], AxiosError>({
    queryKey: ['service-hostnames'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/service-hostnames');
      return response.data;
    },
  });
}

export function useUpdateServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number; data: ServiceHostnameFormData }) => {
      return axios.put(`/api/v1/service-hostnames/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update Service/Hostname.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/service-hostnames/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete Service/Hostname.'), { description: t('Please try again.') });
    },
  });
}

// SNMP receiver queries
export function useCreateSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: SNMPReceiverFormData) => {
      return axios.post('/api/v1/snmp-receivers', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create SNMP receiver.'), { description: t('Please try again.') });
    },
  });
}

export function useGetSNMPReceivers() {
  return useQuery<SNMPReceiver[], AxiosError>({
    queryKey: ['SNMP-receivers'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/snmp-receivers');
      return response.data;
    },
  });
}

export function useUpdateSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, data }: { id: number; data: SNMPReceiverFormData }) => {
      return axios.put(`/api/v1/snmp-receivers/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update SNMP receiver.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/snmp-receivers/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete SNMP receiver.'), { description: t('Please try again.') });
    },
  });
}
