import { z } from 'zod';
import { isThresholdValid, thresholdRegex } from '@/components/alerting/utils';

export const systemCheckFormSchema = z.object({
  check_type: z.literal('system'),
  step1: z.object({
    name: z.string().min(1, 'Check Name Required').max(255, 'Must be less than 255 characters'),
    association: z.object({
      type: z.enum(['cpu', 'memory', 'drive']),
      id: z.string(),
    }),
    metric: z.string(),
  }),
  step2: z
    .object({
      forcecheck: z.boolean().optional(),
      warning_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      critical_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
    })
    .superRefine((data, ctx) => {
      if (!isThresholdValid(data.warning_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['warning_threshold'],
        });
      }
      if (!isThresholdValid(data.critical_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['critical_threshold'],
        });
      }
    }),
  step3: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
});

export const suricataCheckFormSchema = z.object({
  check_type: z.literal('suricata'),
  step1: z
    .object({
      name: z.string().min(1, 'Check Name Required').max(255, 'Must be less than 255 characters'),
      association: z
        .object({
          type: z.literal('signature_id'),
          id: z.string(),
        })
        .superRefine((data, ctx) => {
          if (!data.id) {
            ctx.addIssue({
              code: 'custom',
              message: 'Signature ID Required',
              path: ['id'],
            });
          }
          if (!Number.isInteger(Number(data.id))) {
            ctx.addIssue({
              code: 'custom',
              message: 'Signature ID must be an integer',
              path: ['id'],
            });
          }
          if (parseInt(data.id) <= 0) {
            ctx.addIssue({
              code: 'custom',
              message: 'Signature ID must be a positive number',
              path: ['id'],
            });
          }
        }),
      metric: z.enum(['alert_count']),
      forcecheck: z.boolean().optional(),
      warning_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      critical_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      lookback_period: z.string().min(1, 'Lookback Period Required'),
      check_frequency: z.string().min(1, 'Check Frequency Required'),
    })
    .superRefine((data, ctx) => {
      if (!isThresholdValid(data.warning_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['warning_threshold'],
        });
      }
      if (!isThresholdValid(data.critical_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['critical_threshold'],
        });
      }
    }),
  step2: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
});

export const flowSourceCheckFormSchema = z.object({
  check_type: z.literal('flow_source'),
  step1: z.object({
    name: z.string().min(1, 'Check Name Required').max(255, 'Must be less than 255 characters'),
    association: z
      .object({
        type: z.enum(['source', 'sourcegroup']),
        id: z.string(),
      })
      .superRefine((data, ctx) => {
        if (!data.id) {
          ctx.addIssue({
            code: 'custom',
            message: 'Check Association Required',
            path: ['type'],
          });
          ctx.addIssue({
            code: 'custom',
            message: 'Check Association Required',
            path: ['id'],
          });
        }
      }),
  }),
  step2: z
    .object({
      metric: z.enum(['bytes', 'flows', 'packets', 'pps', 'bps', 'bpp', 'abnormal_behavior']),
      forcecheck: z.boolean().optional(),
      warning_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      critical_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      queries: z.array(
        z.object({
          location: z.enum(['destination', 'source', 'destination_or_source']),
          location_type: z.enum(['port', 'ip', 'network']),
          location_bool: z.enum(['is', 'is_not']),
          location_value: z.string(),
        })
      ),
    })
    .superRefine((data, ctx) => {
      if (data.metric === 'abnormal_behavior') {
        return;
      }

      if (!isThresholdValid(data.warning_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['warning_threshold'],
        });
      }
      if (!isThresholdValid(data.critical_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['critical_threshold'],
        });
      }

      data.queries.forEach((query, index) => {
        if (query.location_type === 'port' && !query.location_value.match(/^\d+$/)) {
          ctx.addIssue({
            code: 'custom',
            message: 'Port must be a number',
            path: ['queries', index, 'location_value'],
          });
        }

        if (
          query.location_type === 'ip' &&
          !query.location_value.match(/^(?:(?:25[0-5]|2[0-4]\d|1?\d{1,2})(?:\.(?!$)|$)){4}$/)
        ) {
          ctx.addIssue({
            code: 'custom',
            message: 'IP must be valid',
            path: ['queries', index, 'location_value'],
          });
        }

        if (
          query.location_type === 'network' &&
          !query.location_value.match(
            /^(?:(?:[1-9]?[0-9]|1[0-9][0-9]|2(?:[0-4][0-9]|5[0-5]))\.){3}(?:[1-9]?[0-9]|1[0-9][0-9]|2(?:[0-4][0-9]|5[0-5]))\/(?:[12]?[0-9]|3[0-2])$/
          )
        ) {
          ctx.addIssue({
            code: 'custom',
            message: 'Network must be valid',
            path: ['queries', index, 'location_value'],
          });
        }
      });
    }),
  step3: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
});

export const nmapCheckFormSchema = z.object({
  check_type: z.literal('nmap'),
  step1: z.object({
    name: z.string().min(1, 'Check Name Required').max(255, 'Must be less than 255 characters'),
    association: z
      .object({
        type: z.enum(['scheduled_scan']),
        id: z.string(),
      })
      .superRefine((data, ctx) => {
        if (!data.id) {
          ctx.addIssue({
            code: 'custom',
            message: 'Check Association Required',
            path: ['type'],
          });
          ctx.addIssue({
            code: 'custom',
            message: 'Check Association Required',
            path: ['id'],
          });
        }
      }),
  }),
  step2: z
    .object({
      metric: z.enum(['ports_open', 'ports_closed']),
      forcecheck: z.boolean().optional(),
      warning_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
      critical_threshold: z
        .string()
        .regex(thresholdRegex, 'Invalid threshold syntax')
        .transform((val) => val ?? ''),
    })
    .superRefine((data, ctx) => {
      if (!isThresholdValid(data.warning_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['warning_threshold'],
        });
      }
      if (!isThresholdValid(data.critical_threshold)) {
        ctx.addIssue({
          code: 'custom',
          message: 'The start of the range must be less than or equal to the end of the range',
          path: ['critical_threshold'],
        });
      }
    }),
  step3: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
});
