import { Link } from 'react-router-dom';
import { CirclePlus } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from '@/components/ui/command';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { X } from 'lucide-react';
import { useGetDashboards } from '@/components/dashboard/queries';
import DeleteDashboardDialog from '@/components/dashboard/DeleteDashboardDialog';
import React, { useState } from 'react';
import AddEditDashboardDialog from '@/components/dashboard/AddEditDashboardDialog';
import { useTranslation } from 'react-i18next';
import { useDashboard } from '@/components/dashboard/DashboardContext';
import { useGetUsers } from '@/components/user/queries';
import { cn } from '@/lib/utils';
import { useAuth } from '@/contexts/AuthContext';

export default function DashboardDropdown({
  children,
  className,
  align = 'center',
}: {
  children: React.ReactNode;
  className?: string;
  align?: 'start' | 'center' | 'end';
}) {
  const [open, setOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<number | undefined>(undefined);
  const { data: dashboards, isLoading } = useGetDashboards();
  const { t } = useTranslation();
  const { dashboard } = useDashboard();
  const { data: users = [], isError: isUsersError } = useGetUsers();
  const { user } = useAuth();

  if (isLoading) return children;
  if (!dashboards || !users || !dashboard) return;

  const title = dashboard ? dashboard.title : '';
  const userId = selectedUser !== undefined ? selectedUser : dashboard.user_id !== -1 ? dashboard.user_id : user?.id;
  const currentDashboard = `${userId}-${title}`;
  const currentUser = users?.find((user) => user.id === userId)?.username;
  const usersAvailable = !isUsersError && users && users.length > 0;

  return (
    <div className={`${className} flex items-center justify-between gap-2`}>
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>{children}</PopoverTrigger>
        <PopoverContent className={cn('grid w-fit grid-cols-2 p-0')} align={align}>
          {usersAvailable && (
            <Command
              defaultValue={currentUser}
              className="bg-background hover:bg-card rounded-tr-none rounded-br-none border-r transition-colors"
            >
              <CommandInput placeholder={t('Search users...')} />
              <CommandList>
                <CommandEmpty>{t('No users found')}.</CommandEmpty>
                <CommandGroup heading={t('Users')}>
                  {users.map((user) => (
                    <CommandItem key={user.id} className="py-0" onSelect={() => setSelectedUser(user.id)} asChild>
                      <div
                        className={cn(
                          'group data-[selected=true]:bg-accent/50 flex h-9 w-full items-center justify-between hover:cursor-pointer',
                          userId === user.id && 'data-[selected=true]:bg-accent bg-accent'
                        )}
                      >
                        <span className="max-w-30 truncate">{user.username}</span>
                      </div>
                    </CommandItem>
                  ))}
                </CommandGroup>
              </CommandList>
            </Command>
          )}
          {/* Dashboard Command */}
          <Command
            defaultValue={currentDashboard}
            className={cn(
              'bg-background hover:bg-card transition-colors',
              usersAvailable ? 'rounded-l-none' : 'col-span-2'
            )}
          >
            <CommandInput placeholder={t('Search dashboards...')} />
            <CommandList>
              <CommandEmpty>{t('No dashboards found')}.</CommandEmpty>
              <AddEditDashboardDialog userId={userId}>
                <div className="hover:bg-secondary mx-1 mt-1 flex h-9 items-center gap-1 rounded-sm px-1 py-4 text-xs hover:cursor-pointer">
                  <Button variant="ghost" className="m-0 h-4 w-4 p-0">
                    <CirclePlus className="text-primary" />
                  </Button>
                  {t('New Dashboard')}
                </div>
              </AddEditDashboardDialog>
              <CommandGroup heading={t('Dashboards')}>
                {dashboards
                  .filter((dashboard) => dashboard.user_id === userId)
                  .map((dashboard) => (
                    <CommandItem key={dashboard.id} className="py-0" value={`${dashboard.user_id}-${dashboard.title}`}>
                      <div className="flex h-9 w-full items-center justify-between">
                        <Link
                          state={{ title: dashboard.title, icon: dashboard.icon, user: selectedUser }}
                          to={'/dashboards/' + dashboard.id}
                          className="flex h-full w-full items-center gap-2"
                          onClick={() => setOpen(false)}
                        >
                          <span className="max-w-30 truncate">{dashboard.title}</span>
                        </Link>
                        <DeleteDashboardDialog id={dashboard.id}>
                          <Button className="text-muted-foreground group m-0 h-4 w-4 bg-transparent hover:bg-transparent">
                            <X className="group-hover:text-white" />
                          </Button>
                        </DeleteDashboardDialog>
                      </div>
                    </CommandItem>
                  ))}
              </CommandGroup>
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>
    </div>
  );
}
