import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import React, { useEffect, useMemo, useState } from 'react';
import { Button } from '@/components/ui/button';
import { useAddDashlet, useEditDashlet } from '@/components/dashboard/dashlet/queries';
import { SubmitHandler, useForm, useFieldArray, useWatch, useController } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form';
import { dashlet, SourceDataOption } from '@/components/dashboard/types';
import { Input } from '@/components/ui/input';
import { Select, SelectItem, SelectTrigger, SelectValue, SelectContent } from '@/components/ui/select';
import { useTranslation } from 'react-i18next';
import { dashletMap } from '@/components/dashboard/dashlet/map';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { useGetSources } from '@/components/source/queries';
import { useGetScheduledScans } from '@/components/nmap/scheduledscans/queries';
import { scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { Plus, Trash, Droplet, Droplets, ListOrdered, Radar } from 'lucide-react';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Badge } from '@/components/ui/badge';
import { Source } from '@/components/source/types';
import { ToggleGroup, ToggleGroupItem } from '@/components/ui/toggle-group';
import { useDashboard } from '@/components/dashboard/DashboardContext';
import { Slider } from '@/components/ui/slider';
import { useGetNmapVersion } from '@/components/nmap/queries';
import NmapNotInstalled from '@/components/nmap/NmapNotInstalled';

const compatabilityMatrix: Record<string, Record<string, 0 | 1>> = {
  area: {
    source: 1,
    sources: 1,
    talkers: 0,
    scheduledScan: 1,
  },
  bar: {
    source: 1,
    sources: 1,
    talkers: 1,
    scheduledScan: 1,
  },
  line: {
    source: 1,
    sources: 1,
    talkers: 0,
    scheduledScan: 1,
  },
  pie: {
    source: 0,
    sources: 0,
    talkers: 1,
    scheduledScan: 0,
  },
  radial: {
    source: 0,
    sources: 0,
    talkers: 1,
    scheduledScan: 0,
  },
  table: {
    source: 0,
    sources: 0,
    talkers: 1,
  },
};

type AddEditDashletFormProps = {
  dashboardId?: number;
  dashlet?: dashlet;
  setOpen: (open: boolean) => void;
  sources: Source[];
  scheduledScans: scheduledNmapScan[];
};

function AddEditDashletForm({ dashboardId, dashlet, setOpen, sources, scheduledScans }: AddEditDashletFormProps) {
  const { t } = useTranslation();
  const { range } = useDashboard().dashboard;
  const addDashletMutation = useAddDashlet();
  const editDashletMutation = useEditDashlet();
  const { data: nmapVersion, isLoading: nmapVersionLoading } = useGetNmapVersion();
  const nmapNotInstalled = (!nmapVersion || nmapVersion.nmap_version === 'None') && !nmapVersionLoading;

  const dataMap = useMemo(
    () => ({
      source: { icon: Droplet, title: t('Source') },
      sources: { icon: Droplets, title: t('Sources') },
      talkers: { icon: ListOrdered, title: t('Top Talkers') },
      scheduledScan: { icon: Radar, title: t('Nmap') },
    }),
    [t]
  );

  const metrics: Record<string, string> = {
    bytes: t('Bytes'),
    flows: t('Flows'),
    packets: t('Packets'),
    bps: t('Bits/Seconds'),
  };

  const scheduledScanMetrics: Record<string, string> = {
    hosts_up: t('Hosts Up'),
    ports_open: t('Ports Open'),
    ports_closed: t('Ports Closed'),
    ports_filtered: t('Ports Filtered'),
  };

  const endpoints: Record<string, string> = {
    srcip: t('Source IP'),
    dstip: t('Destination IP'),
    srcport: t('Source Port'),
    dstport: t('Destination Port'),
  };

  const form = useForm<dashlet>({
    defaultValues: {
      title: dashlet ? dashlet.title : t('New Dashlet'),
      description: dashlet ? dashlet.description : t('New Description'),
      data_options: dashlet
        ? dashlet.data_options
        : {
            type: 'source',
            multi_source_data_options: {
              metric: 'bytes',
              sources: [
                {
                  color: 1,
                  source_id: sources.length > 0 ? sources[0].id : 0,
                  name: sources.length > 0 ? `${sources[0].name}` : '',
                },
              ],
            },
            single_source_data_options: {
              source_id: sources.length > 0 ? sources[0].id : 0,
              metrics: Object.keys(metrics),
            },
            top_talker_data_options: {
              source_id: sources.length > 0 ? sources[0].id : 0,
              num_talkers: [5],
              order_by: 'bytes',
              group_by: 'srcip',
            },
            scheduled_scan_data_options: {
              scheduled_scan_id: scheduledScans.length > 0 ? scheduledScans[0].id : 0,
              metrics: Object.keys(scheduledScanMetrics),
            },
          },
      vis_options: dashlet
        ? dashlet.vis_options
        : {
            type: 'area',
            area_options: {
              type: 'natural',
              legend: true,
              stacked: 'no',
              gradient: true,
            },
          },
      dashboard_id: dashlet ? dashlet.dashboard_id : dashboardId,
      width: dashlet ? dashlet.width : 35,
      height: dashlet ? dashlet.height : 20,
      x: dashlet ? dashlet.x : 0,
      y: dashlet ? dashlet.y : 0,
    },
  });

  const { fields, prepend, remove } = useFieldArray({
    control: form.control,
    name: 'data_options.multi_source_data_options.sources',
  });

  const compatabilityWatch = useWatch({ control: form.control, name: ['data_options.type', 'vis_options.type'] });
  const { field: dataOptionsField } = useController({ control: form.control, name: 'data_options.type' });

  useEffect(() => {
    const visibleTabs = Object.entries(dataMap).filter(([key]) => compatabilityMatrix[compatabilityWatch[1]][key]);
    const selectedIsVisible = visibleTabs.some(([key]) => key === dataOptionsField.value);
    if (!selectedIsVisible && visibleTabs.length > 0 && dataOptionsField.value !== visibleTabs[0][0]) {
      dataOptionsField.onChange(visibleTabs[0][0]);
    }
  }, [dataMap, compatabilityWatch, dataOptionsField]);

  function prependSource() {
    const usedIds = new Set(fields.map((item) => Number(item.source_id)));
    const remainingItems = sources.filter((item) => !usedIds.has(item.id));

    if (remainingItems.length === 0) return;

    const nextItem = remainingItems[0];
    prepend({
      source_id: nextItem.id,
      name: nextItem.name,
      color: fields.length > 0 ? (Number(fields[0].color) + 1) % 10 : 1,
    });
  }

  const onSubmit: SubmitHandler<dashlet> = (data) => {
    if (dashlet) {
      editDashletMutation.mutate({ data: { ...data, id: dashlet.id } });
    } else {
      addDashletMutation.mutate(data);
    }
    setOpen(false);
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)}>
        <div className="grid gap-2 pb-4 xl:grid-cols-4 xl:gap-4">
          <div className="col-span-3 grid h-full gap-2 pr-4 xl:border-r">
            <FormField
              control={form.control}
              name="vis_options.type"
              render={({ field }) => (
                <Tabs onValueChange={field.onChange} defaultValue={field.value}>
                  {Object.entries(dashletMap).map(([key]) => {
                    const Component = dashletMap[key].component;
                    return (
                      <TabsContent key={key} value={key} className="max-h-70 min-h-70 max-w-[817.5px] overflow-y-auto">
                        {form.watch('data_options.type') === 'scheduledScan' && nmapNotInstalled ? (
                          <NmapNotInstalled />
                        ) : (
                          <Component {...form.watch('data_options')} range={range} />
                        )}
                      </TabsContent>
                    );
                  })}
                  <FormControl>
                    <FormItem>
                      <FormLabel>
                        <Badge variant="blue">1</Badge>
                        {t('Choose your visualization')}
                      </FormLabel>
                      <TabsList className="space-x-0">
                        {Object.entries(dashletMap).map(([key, value]) => (
                          <TabsTrigger key={key} value={key} className="gap-1">
                            {value.icon ? <value.icon className="size-4" /> : null}
                            {value.title}
                          </TabsTrigger>
                        ))}
                      </TabsList>
                    </FormItem>
                  </FormControl>
                </Tabs>
              )}
            />
            <div className="border-border w-full border-b" />
            <FormField
              control={form.control}
              name="data_options.type"
              render={({ field }) => (
                <Tabs onValueChange={field.onChange} value={field.value}>
                  <FormControl>
                    <FormItem>
                      <FormLabel>
                        <Badge variant="blue">2</Badge>
                        {t('Choose your data')}
                      </FormLabel>

                      <TabsList className="space-x-0">
                        {Object.entries(dataMap).map(([key, value]) => {
                          const visible = compatabilityMatrix[compatabilityWatch[1]][key];

                          return (
                            <TabsTrigger
                              key={key}
                              value={key}
                              className={`gap-1 transition-all duration-300 ${
                                visible ? 'max-w-50 opacity-100' : 'pointer-events-none max-w-0 px-0 opacity-0'
                              }`}
                            >
                              {value.icon ? <value.icon className="size-4" /> : null}
                              <span>{value.title}</span>
                            </TabsTrigger>
                          );
                        })}
                      </TabsList>
                    </FormItem>
                  </FormControl>
                </Tabs>
              )}
            />
            <div className="border-border w-full border-b" />
            {form.watch('data_options.type') === 'sources' && (
              <div className="space-y-2">
                <FormLabel>
                  <Badge variant="blue">3</Badge>
                  {t('Configure your sources')}
                </FormLabel>
                <div className="bg-card border-border space-y-2 rounded-lg border p-4">
                  <div className="flex items-end justify-between">
                    <FormField
                      control={form.control}
                      name="data_options.multi_source_data_options.metric"
                      render={({ field }) => (
                        <Tabs onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <FormItem>
                              <TabsList>
                                {Object.entries(metrics).map(([key, value]) => (
                                  <TabsTrigger key={key} value={key}>
                                    {value}
                                  </TabsTrigger>
                                ))}
                              </TabsList>
                            </FormItem>
                          </FormControl>
                        </Tabs>
                      )}
                    />
                    <Button
                      size="sm"
                      variant="outline"
                      type="button"
                      disabled={fields.length >= sources.length}
                      onClick={prependSource}
                    >
                      <Plus /> {t('Add Source')}
                    </Button>
                  </div>
                  {fields.map((field, index) => (
                    <div key={field.id} className="bg-background flex items-center gap-2 rounded-lg p-1">
                      <FormField
                        control={form.control}
                        name={`data_options.multi_source_data_options.sources.${index}.color`}
                        render={({ field }) => (
                          <div
                            className="mx-2 size-4 rounded-full"
                            style={{ backgroundColor: `var(--chart-${field.value}` }}
                          />
                        )}
                      />
                      <FormField
                        control={form.control}
                        name={`data_options.multi_source_data_options.sources.${index}.source_id`}
                        render={({ field }) => (
                          <FormItem>
                            <Select
                              onValueChange={(value) => {
                                const selectedSource = sources.find((source) => `${source.id}` === value);
                                field.onChange(value);
                                if (selectedSource) {
                                  form.setValue(
                                    `data_options.multi_source_data_options.sources.${index}.name`,
                                    selectedSource.name
                                  );
                                }
                              }}
                              defaultValue={`${field.value}`}
                            >
                              <FormControl>
                                <SelectTrigger>
                                  <SelectValue placeholder={t('Select source')} />
                                </SelectTrigger>
                              </FormControl>
                              <SelectContent>
                                {sources.map((source) => (
                                  <SelectItem
                                    key={source.id}
                                    value={`${source.id}`}
                                    disabled={form
                                      .watch('data_options.multi_source_data_options.sources')
                                      ?.some(
                                        (opt: SourceDataOption, i) =>
                                          `${opt.source_id}` === `${source.id}` && i !== index
                                      )}
                                  >
                                    {source.name}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </FormItem>
                        )}
                      />
                      <span>{metrics[form.watch('data_options.multi_source_data_options.metric') ?? '']}</span>
                      <Button
                        className="ml-auto"
                        type="button"
                        size="icon"
                        variant="ghost"
                        onClick={() => remove(index)}
                      >
                        <Trash />
                      </Button>
                    </div>
                  ))}
                </div>
              </div>
            )}
            {form.watch('data_options.type') === 'source' && (
              <div className="space-y-2">
                <FormLabel>
                  <Badge variant="blue">3</Badge>
                  {t('Configure your source')}
                </FormLabel>
                <div className="bg-card border-border space-y-2 rounded-lg border p-4">
                  <FormField
                    control={form.control}
                    name={`data_options.single_source_data_options.source_id`}
                    render={({ field }) => (
                      <FormItem className="w-fit">
                        <Select onValueChange={field.onChange} defaultValue={`${field.value}`}>
                          <FormControl>
                            <SelectTrigger className="focus:ring-inset">
                              <SelectValue placeholder={t('Select source')} />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {sources.map((source) => (
                              <SelectItem key={source.id} value={`${source.id}`}>
                                {source.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name={`data_options.single_source_data_options.metrics`}
                    render={({ field }) => (
                      <FormItem>
                        <ToggleGroup
                          className="gap-1"
                          type="multiple"
                          value={field.value}
                          onValueChange={field.onChange}
                          defaultValue={Object.keys(metrics)}
                        >
                          {Object.entries(metrics).map(([key, value], index) => (
                            <ToggleGroupItem key={key} value={key} className="min-w-fit rounded-lg border">
                              <div
                                className="size-4 rounded-full"
                                style={{ backgroundColor: `var(--chart-${index + 1})` }}
                              />
                              {value}
                            </ToggleGroupItem>
                          ))}
                        </ToggleGroup>
                      </FormItem>
                    )}
                  />
                </div>
              </div>
            )}
            {form.watch('data_options.type') === 'talkers' && (
              <div className="space-y-2">
                <FormLabel>
                  <Badge variant="blue">3</Badge>
                  {t('Configure your data')}
                </FormLabel>
                <div className="border-border bg-card w-full space-y-2 rounded-lg border p-4">
                  <FormField
                    control={form.control}
                    name={`data_options.top_talker_data_options.source_id`}
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-4">
                        <FormLabel>{t('Source')}</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={`${field.value}`}>
                          <FormControl>
                            <SelectTrigger className="focus:ring-inset">
                              <SelectValue placeholder={t('Select source')} />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {sources.map((source) => (
                              <SelectItem key={source.id} value={`${source.id}`}>
                                {source.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name={`data_options.top_talker_data_options.num_talkers`}
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-4">
                        <FormLabel># {t('Talkers')}</FormLabel>
                        <FormControl>
                          <Slider
                            value={field.value}
                            onValueChange={field.onChange}
                            defaultValue={[5]}
                            min={1}
                            max={20}
                            step={1}
                          />
                        </FormControl>
                        {field.value}
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="data_options.top_talker_data_options.order_by"
                    render={({ field }) => (
                      <Tabs onValueChange={field.onChange} defaultValue={field.value}>
                        <FormControl>
                          <FormItem className="grid grid-cols-4">
                            <FormLabel>{t('Order by')}</FormLabel>
                            <TabsList>
                              {Object.entries(metrics).map(
                                ([key, value]) =>
                                  key !== 'bps' && (
                                    <TabsTrigger key={key} value={key}>
                                      {value}
                                    </TabsTrigger>
                                  )
                              )}
                            </TabsList>
                          </FormItem>
                        </FormControl>
                      </Tabs>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="data_options.top_talker_data_options.group_by"
                    render={({ field }) => (
                      <Tabs onValueChange={field.onChange} defaultValue={field.value}>
                        <FormControl>
                          <FormItem className="grid grid-cols-4">
                            <FormLabel>{t('Group by')}</FormLabel>
                            <TabsList>
                              {Object.entries(endpoints).map(([key, value]) => (
                                <TabsTrigger key={key} value={key}>
                                  {value}
                                </TabsTrigger>
                              ))}
                            </TabsList>
                          </FormItem>
                        </FormControl>
                      </Tabs>
                    )}
                  />
                </div>
              </div>
            )}
            {form.watch('data_options.type') === 'scheduledScan' && (
              <div className="space-y-2">
                <FormLabel>
                  <Badge variant="blue">3</Badge>
                  {t('Configure your scheduled scan')}
                </FormLabel>
                <div className="bg-card border-border space-y-2 rounded-lg border p-4">
                  <FormField
                    control={form.control}
                    name={`data_options.scheduled_scan_data_options.scheduled_scan_id`}
                    render={({ field }) => (
                      <FormItem className="w-fit">
                        <Select onValueChange={field.onChange} defaultValue={`${field.value}`}>
                          <FormControl>
                            <SelectTrigger className="focus:ring-inset">
                              <SelectValue placeholder={t('Select scheduled scan')} />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {scheduledScans.map((scheduledScan) => (
                              <SelectItem key={scheduledScan.id} value={`${scheduledScan.id}`}>
                                {scheduledScan.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name={`data_options.scheduled_scan_data_options.metrics`}
                    render={({ field }) => (
                      <FormItem>
                        <ToggleGroup
                          className="gap-1"
                          type="multiple"
                          value={field.value}
                          onValueChange={field.onChange}
                          defaultValue={Object.keys(scheduledScanMetrics)}
                        >
                          {Object.entries(scheduledScanMetrics).map(([key, value], index) => (
                            <ToggleGroupItem key={key} value={key} className="min-w-fit rounded-lg border">
                              <div
                                className="size-4 rounded-full"
                                style={{ backgroundColor: `var(--chart-${index + 1})` }}
                              />
                              {value}
                            </ToggleGroupItem>
                          ))}
                        </ToggleGroup>
                      </FormItem>
                    )}
                  />
                </div>
              </div>
            )}
          </div>
          <div className="col-span-3 xl:col-span-1">
            <div className="border-border mb-2 w-full border-b xl:hidden" />
            <div className="space-y-4">
              <FormLabel>
                <Badge variant="blue">4</Badge>
                {t('Customize your dashlet')}
              </FormLabel>
              <div className="grid grid-cols-2 flex-col gap-2 xl:flex">
                <FormField
                  control={form.control}
                  name="title"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t('Title')}</FormLabel>
                      <FormControl>
                        <Input placeholder="New Dashlet" {...field} className="focus:ring-inset" />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t('Description')}</FormLabel>
                      <FormControl>
                        <Input placeholder="New Description" {...field} className="focus:ring-inset" />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </div>
            </div>
          </div>
        </div>
        <DialogFooter className="bg-background absolute right-0 bottom-0 z-10 w-[calc(100%+1rem)] border-t px-4 pt-4">
          <Button type="submit">{t('Submit')}</Button>
        </DialogFooter>
      </form>
    </Form>
  );
}

export function AddDashletDialog({ dashboardId, children }: { dashboardId: number; children: React.ReactNode }) {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();
  const { data: sources } = useGetSources();
  const { data: scheduledScans } = useGetScheduledScans();

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>{children}</DialogTrigger>
      <DialogContent className="flex h-6/7 w-3/5 min-w-3/5 flex-col p-4">
        <DialogHeader>
          <DialogTitle>{t('Add Dashlet')}</DialogTitle>
          <DialogDescription>{t("Choose your dashlet options here. Click submit when you're done.")}</DialogDescription>
        </DialogHeader>
        <ScrollArea className="-mr-4 h-[calc(100%-4rem)] pr-4 pb-12">
          <AddEditDashletForm
            dashboardId={dashboardId}
            setOpen={setOpen}
            sources={sources ?? []}
            scheduledScans={scheduledScans ?? []}
          />
        </ScrollArea>
      </DialogContent>
    </Dialog>
  );
}

export function EditDashletDialog({ dashlet, setOpen }: { dashlet: dashlet; setOpen: (open: boolean) => void }) {
  const { t } = useTranslation();
  const { data: sources } = useGetSources();
  const { data: scheduledScans } = useGetScheduledScans();

  return (
    <DialogContent className="flex h-6/7 w-3/5 min-w-3/5 flex-col p-4">
      <DialogHeader>
        <DialogTitle>{'Edit Dashlet'}</DialogTitle>
        <DialogDescription>{t("Choose your dashlet options here. Click save when you're done.")}</DialogDescription>
      </DialogHeader>
      <ScrollArea className="-mr-4 h-[calc(100%-4rem)] pr-4 pb-12">
        <AddEditDashletForm
          dashlet={dashlet}
          setOpen={setOpen}
          sources={sources ?? []}
          scheduledScans={scheduledScans ?? []}
        />
      </ScrollArea>
    </DialogContent>
  );
}
