'use client';

import { Pie, PieChart } from 'recharts';
import { DashletDataOptions } from '../types';
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from '@/components/ui/chart';
import { PieChart as Icon, LoaderCircle } from 'lucide-react';
import { useGetSourceTopTalkersData } from './queries';
import { useMemo } from 'react';
import { useTranslation } from 'react-i18next';
import NoDataAlert from '@/components/dashboard/dashlet/NoDataAlert';

export const meta = {
  name: 'pie',
  title: 'Pie Chart',
  description: 'A simple pie chart',
  icon: Icon,
};

const defaultTalkerOptions = {
  source_id: 0,
  num_talkers: [5],
  order_by: 'bytes',
  group_by: 'srcip',
};

export default function ChartPie(props: DashletDataOptions) {
  const { t } = useTranslation();
  const {
    data: talkersData,
    isLoading: talkersLoading,
    isError: talkersError,
  } = useGetSourceTopTalkersData(
    Number(props.top_talker_data_options?.source_id ?? 0),
    props.range,
    props.top_talker_data_options ?? defaultTalkerOptions,
    { enabled: props.type === 'talkers' }
  );

  const data = useMemo(() => (props.type === 'talkers' ? talkersData : {}), [props.type, talkersData]);

  const isLoading = props.type === 'talkers' ? talkersLoading : false;
  const isError = props.type === 'talkers' ? talkersError : false;
  const dataKey = props.type === 'talkers' ? (props.top_talker_data_options?.order_by ?? 'bytes') : 'bytes';
  const nameKey = props.type === 'talkers' ? (props.top_talker_data_options?.group_by ?? 'srcip') : 'srcip';

  const [chartData, chartConfig] = useMemo(() => {
    const metrics: Record<string, string> = {
      bytes: t('Bytes'),
      flows: t('Flows'),
      packets: t('Packets'),
      bps: t('Bits/Seconds'),
    };

    if (!data || !data[nameKey]) return [[], { [dataKey]: { label: metrics[dataKey] } }];

    const sliced = data[nameKey].slice(1);
    const config: ChartConfig = {
      [dataKey]: { label: metrics[dataKey] },
    };

    const dataPoints = sliced.map((flow: { [key: string]: string }, index: number) => {
      const color = `var(--chart-${((index + Math.floor(index / 5)) % 5) + 1})`;
      config[flow[nameKey]] = {
        label: flow[nameKey],
        color,
      };
      return {
        [nameKey]: flow[nameKey],
        [dataKey]: Number(flow[dataKey]),
        fill: color,
      };
    });

    return [dataPoints, config];
  }, [t, data, dataKey, nameKey]);

  if (isError) return <div>{t('Data not found')}</div>;
  if (isLoading || !data)
    return (
      <div className="flex size-full items-center justify-center">
        <LoaderCircle className="animate-spin" />
      </div>
    );

  if (chartData.length === 0) return <NoDataAlert />;

  return (
    <ChartContainer config={chartConfig} className="mx-auto aspect-square h-full w-full">
      <PieChart>
        <ChartTooltip cursor={false} content={<ChartTooltipContent hideLabel />} />
        <Pie data={chartData} dataKey={dataKey} nameKey={nameKey} isAnimationActive={!props.isExporting} />
        <ChartLegend
          content={<ChartLegendContent nameKey={nameKey} />}
          className="-translate-y-2 flex-wrap gap-2 *:basis-1/4 *:justify-center"
        />
      </PieChart>
    </ChartContainer>
  );
}
