'use client';

import { RadialBar, RadialBarChart } from 'recharts';
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from '@/components/ui/chart';
import { Radar as Icon, LoaderCircle } from 'lucide-react';
import { DashletDataOptions } from '@/components/dashboard/types';
import { useMemo } from 'react';
import { useTranslation } from 'react-i18next';
import { useGetSourceTopTalkersData } from '@/components/dashboard/dashlets/queries';
import NoDataAlert from '@/components/dashboard/dashlet/NoDataAlert';

export const meta = {
  name: 'radial',
  title: 'Radial Chart',
  description: 'A simple radial chart',
  icon: Icon,
};

const defaultTalkerOptions = {
  source_id: 0,
  num_talkers: [5],
  order_by: 'bytes',
  group_by: 'srcip',
};

export default function ChartRadial(props: DashletDataOptions) {
  const { t } = useTranslation();
  const {
    data: talkersData,
    isLoading: talkersLoading,
    isError: talkersError,
  } = useGetSourceTopTalkersData(
    Number(props.top_talker_data_options?.source_id ?? 0),
    props.range,
    props.top_talker_data_options ?? defaultTalkerOptions,
    { enabled: props.type === 'talkers' }
  );

  const data = useMemo(() => (props.type === 'talkers' ? talkersData : {}), [props.type, talkersData]);

  const isLoading = props.type === 'talkers' ? talkersLoading : false;
  const isError = props.type === 'talkers' ? talkersError : false;
  const dataKey = props.type === 'talkers' ? (props.top_talker_data_options?.order_by ?? 'bytes') : 'bytes';
  const nameKey = props.type === 'talkers' ? (props.top_talker_data_options?.group_by ?? 'srcip') : 'srcip';

  const [chartData, chartConfig] = useMemo(() => {
    const metrics: Record<string, string> = {
      bytes: t('Bytes'),
      flows: t('Flows'),
      packets: t('Packets'),
      bps: t('Bits/Seconds'),
    };

    if (!data || !data[nameKey]) return [[], { [dataKey]: { label: metrics[dataKey] } }];

    const sliced = data[nameKey].slice(1);
    const config: ChartConfig = {
      [dataKey]: { label: metrics[dataKey] },
    };

    const dataPoints = sliced.map((flow: { [key: string]: string }, index: number) => {
      const color = `var(--chart-${(index % 5) + 1})`;
      config[flow[nameKey]] = {
        label: flow[nameKey],
        color,
      };
      return {
        [nameKey]: flow[nameKey],
        [dataKey]: Number(flow[dataKey]),
        fill: color,
      };
    });

    return [dataPoints, config];
  }, [t, data, dataKey, nameKey]);

  if (isError) return <div>{t('Source not found')}</div>;
  if (isLoading || !data)
    return (
      <div className="flex size-full items-center justify-center">
        <LoaderCircle className="animate-spin" />
      </div>
    );

  if (chartData.length === 0) return <NoDataAlert />;

  const legendRows = Math.ceil(chartData.length / 4);
  const cy = `${45 - legendRows * 5}%`;

  return (
    <ChartContainer config={chartConfig} className="mx-auto aspect-square h-full w-full">
      <RadialBarChart data={chartData} innerRadius={'30%'} margin={{ top: 0, right: 0, bottom: 0, left: 0 }} cy={cy}>
        <ChartTooltip cursor={false} content={<ChartTooltipContent hideLabel nameKey={nameKey} />} />
        <RadialBar dataKey={dataKey} background isAnimationActive={!props.isExporting} />
        <ChartLegend
          content={<ChartLegendContent nameKey={nameKey} />}
          className="-translate-y-2 flex-wrap gap-2 *:basis-1/4 *:justify-center"
        />
      </RadialBarChart>
    </ChartContainer>
  );
}
