import React, { useCallback, useEffect, useRef, useState } from 'react';
import {
  CommandDialog,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
  CommandSeparator,
} from '@/components/ui/command';
import { CommandLoading } from 'cmdk';
import { Link, useNavigate } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { TFunction } from 'i18next';
import { SuggestData, SearchDataType, SuggestReturn } from '@/components/header/types';
import { useSearchSuggest } from '@/components/header/queries';
import routes, { CustomRouteObject } from '@/routes';
import { useQueryClient } from '@tanstack/react-query';
import { useAuth } from '@/contexts/AuthContext';
import { isAdmin } from '../role/utils';
import { useGetIntegrations } from '../queries';

function flattenRoutes(routes: CustomRouteObject[], isAdmin?: boolean): CustomRouteObject[] {
  let result: CustomRouteObject[] = [];

  routes.forEach((route) => {
    if (route.adminOnly && !isAdmin) return;
    if (route.path) result.push(route);
    if (route.children && route.children.length > 0) {
      result = result.concat(flattenRoutes(route.children, isAdmin));
    }
  });

  return result;
}

function addSearchElement(searchData: SearchDataType, element: SuggestReturn, t: TFunction<'translation', undefined>) {
  if (!Object.prototype.hasOwnProperty.call(searchData, t(element.category))) {
    searchData[t(element.category)] = [];
  }
  searchData[t(element.category)].push({
    url: element.url,
    value: t(element.value),
    label: t(element.label),
  });
  return searchData;
}

export const Search = () => {
  const queryClient = useQueryClient();
  const [open, setOpen] = useState(false);
  const [value, setValue] = useState('');
  const { data } = useGetIntegrations();
  const { t } = useTranslation();
  const navigate = useNavigate();
  const { user } = useAuth();
  const scrollRef = useRef<HTMLDivElement>(null);
  const shouldScrollRef = useRef(true);

  const flattenedRoutes = flattenRoutes(routes, isAdmin(user?.role));

  const adjustedRoutes = flattenedRoutes.map((route) => {
    if (route.path === '/wireshark/ringbuffer') {
      return { ...route, isSearchable: data?.ringbuffer && data?.wireshark };
    }
    return route;
  });

  const searchData: SearchDataType = {};
  adjustedRoutes.forEach((route) => {
    if (route.isSearchable && route.path && route.label && route.category) {
      addSearchElement(
        searchData,
        {
          url: route.path,
          value: route.value !== undefined ? t(route.value) : t(route.label),
          label: route.label,
          category: route.category,
        },
        t
      );
    }
  });

  // Search backend for pages and parse it
  const suggestQuery = useSearchSuggest(value);
  if (suggestQuery.isSuccess) {
    suggestQuery.data.forEach((element) => {
      addSearchElement(searchData, element, t);
    });
  }

  const handleOpenDialog = useCallback(
    (e: KeyboardEvent) => {
      if (e.key === 'k' && (e.metaKey || e.ctrlKey)) {
        e.preventDefault();
        setOpen((prev) => !prev);
      }
    },
    [setOpen]
  );

  useEffect(() => {
    if (open) {
      queryClient.invalidateQueries({ queryKey: ['search'] });
      queryClient.invalidateQueries({ queryKey: ['installed_integrations'] });
    }
    document.addEventListener('keydown', handleOpenDialog);
    return () => {
      document.removeEventListener('keydown', handleOpenDialog);
    };
  }, [open, queryClient, handleOpenDialog]);

  return (
    <>
      <div className="relative flex items-center">
        <input
          type="text"
          placeholder={t('Search {{NNA}}', { NNA: 'NNA' })}
          onFocus={() => setOpen(true)}
          id="header-search-box"
          className="text-muted-foreground hover:bg-muted/70 hover:text-accent-foreground focus-visible:ring-ring relative h-8 w-full justify-start rounded-md border bg-transparent px-4 py-2 text-sm font-medium shadow-sm transition-colors focus-visible:ring-1 focus-visible:outline-none"
        />
        <kbd className="bg-muted text-muted-foreground pointer-events-none absolute top-1/2 right-2 inline-flex h-5 -translate-y-1/2 items-center gap-1 rounded border px-1.5 font-mono text-[10px] font-medium opacity-100 select-none">
          {window.navigator.userAgent.includes('Macintosh') ? (
            <>
              <span className="text-xs">⌘</span>K
            </>
          ) : (
            <span className="text-xs">Ctrl K</span>
          )}
        </kbd>
      </div>
      <CommandDialog open={open} onOpenChange={setOpen}>
        <CommandInput
          placeholder={t('Search for a page...')}
          value={value}
          onKeyDown={(event) => {
            if (event.key === 'ArrowDown' || event.key === 'ArrowUp') {
              shouldScrollRef.current = true;
            } else {
              shouldScrollRef.current = false;
            }
          }}
          onValueChange={(input) => setValue(input)}
        />
        <CommandList
          ref={scrollRef}
          className="h-80"
          onWheel={() => (shouldScrollRef.current = true)}
          onScroll={() => {
            if (!shouldScrollRef.current) {
              // Taken from the scrollSelectedIntoView() function from https://github.com/pacocoursey/cmdk/blob/main/cmdk/src/index.tsx
              const item = scrollRef.current?.querySelector('[cmdk-item=""][aria-selected="true"]');
              if (item) {
                if (item.parentElement?.firstChild === item) {
                  // First item in Group, ensure heading is in view
                  item
                    .closest('[cmdk-group=""]')
                    ?.querySelector('[cmdk-group-heading=""]')
                    ?.scrollIntoView({ block: 'nearest' });
                }
                // Ensure the item is always in view
                item.scrollIntoView({ block: 'nearest' });
              }
            }
            shouldScrollRef.current = true;
          }}
        >
          {suggestQuery.isLoading && <CommandLoading />}
          <CommandEmpty>{suggestQuery.isLoading ? '' : t('No results found.')}</CommandEmpty>
          {Object.entries(searchData).map(([key, value]: [string, Array<SuggestData>], index, arr) => {
            return (
              <React.Fragment key={key}>
                <CommandGroup heading={key}>
                  {value.map((element: SuggestData) => {
                    return (
                      <Link key={element.url} className="w-full" to={element.url} onClick={() => setOpen(false)}>
                        <div className="rounded-sm">
                          <CommandItem
                            className="flex gap-2 hover:cursor-pointer"
                            value={element.value}
                            onSelect={() => {
                              setOpen(false);
                              const url = new URL(element.url, window.location.origin);
                              navigate(url.pathname + url.search + url.hash);
                            }}
                          >
                            <span>{element.label}</span>
                          </CommandItem>
                        </div>
                      </Link>
                    );
                  })}
                </CommandGroup>
                {index !== arr.length - 1 && <CommandSeparator className="mb-2" />}
              </React.Fragment>
            );
          })}
        </CommandList>
      </CommandDialog>
    </>
  );
};
