import { useState } from 'react';
import { useForm } from 'react-hook-form';
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetFooter,
  SheetTitle,
  SheetDescription,
  SheetClose,
} from '@/components/ui/sheet';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Form, FormItem, FormControl, FormField, FormLabel, FormMessage } from '@/components/ui/form';
import { Textarea } from '@/components/ui/textarea';
import type { nmapProfile } from './types';
import { useUpdateProfile } from '@/components/nmap/profiles/queries';
import { useTranslation } from 'react-i18next';
import { validateScanHasInterfaceFlag, validateScanParameters } from '@/components/nmap/utils';

export default function EditProfileSheet({
  open,
  onOpenChange,
  profile,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  profile: nmapProfile;
}) {
  const { t } = useTranslation();
  const updateProfileMutation = useUpdateProfile();
  const [error, setError] = useState<string | null>(null);

  const form = useForm<Partial<nmapProfile>>({
    values: {
      name: profile?.name || '',
      parameters: profile?.parameters || '',
      description: profile?.description || '',
      tags: Array.isArray(profile?.tags) ? profile.tags.sort() : [],
    },
  });

  const handleOpenChange = (newOpen: boolean) => {
    if (!newOpen) {
      setError(null);
      form.reset(profile);
    }
    onOpenChange(newOpen);
  };

  const onSubmit = (data: Partial<nmapProfile>) => {
    if (!validateScanParameters(data.parameters || '', setError)) return;
    if (!validateScanHasInterfaceFlag(data.parameters || '', setError, true)) return;
    const id = (data as Partial<nmapProfile>).id ?? profile?.id;
    const payload: Partial<nmapProfile> & { id: number } = {
      ...(data || {}),
      id,
    };
    updateProfileMutation.mutate(payload, {
      onSuccess: () => onOpenChange(false),
    });
  };

  return (
    <Sheet open={open} onOpenChange={handleOpenChange}>
      <SheetContent side="right">
        <SheetHeader>
          <SheetTitle>{t('Edit Profile')}</SheetTitle>
          <SheetDescription>{t('Update the fields below and click Save to update the profile')}.</SheetDescription>
        </SheetHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col gap-4 py-4">
            {error && <div className="text-destructive mb-2 text-sm">{error}</div>}
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Name')}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="parameters"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Parameters')}</FormLabel>
                  <FormControl>
                    <Textarea className="min-h-64 resize-none" {...field} value={field.value ?? ''} rows={3} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="description"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Description')}</FormLabel>
                  <FormControl>
                    <Textarea className="min-h-48 resize-none" {...field} value={field.value ?? ''} rows={3} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="tags"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Comma Separated Tags')}</FormLabel>
                  <FormControl>
                    <Input
                      {...field}
                      value={Array.isArray(field.value) ? field.value.join(', ') : field.value || ''}
                      onChange={(e) => field.onChange(e.target.value)}
                      onBlur={() => {
                        const currentValue = form.getValues('tags');
                        const stringValue = Array.isArray(currentValue) ? currentValue.join(', ') : currentValue || '';
                        const tagsArray = stringValue
                          .split(',')
                          .map((t) => t.trim())
                          .filter(Boolean);
                        field.onChange(tagsArray);
                      }}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <SheetFooter>
              <SheetClose>
                <Button variant="outline" type="button">
                  {t('Cancel')}
                </Button>
              </SheetClose>
              <Button type="submit" disabled={updateProfileMutation.isPending}>
                {updateProfileMutation.isPending ? t('Saving') + '...' : t('Save')}
              </Button>
            </SheetFooter>
          </form>
        </Form>
      </SheetContent>
    </Sheet>
  );
}
