import { Button } from '@/components/ui/button';
import { DataTable, DataTableRowAction } from '@/components/ui/DataTable/DataTable';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import { columns } from '@/components/nmap/profiles/columns';
import AddProfileSheet from '@/components/nmap/profiles/AddProfileSheet';
import { nmapProfile } from '@/components/nmap/profiles/types';
import { Plus } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';
import DeleteProfileDialog from '@/components/nmap/profiles/DeleteProfileDialog';
import { useState } from 'react';

type ProfilesProps = {
  data: nmapProfile[] | undefined;
  isLoading: boolean;
};

export default function Profiles({ data, isLoading }: ProfilesProps) {
  const { t } = useTranslation();
  const { user } = useAuth();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [ids, setIds] = useState<number[]>([]);

  const actions: DataTableRowAction<nmapProfile>[] = [];

  if (hasPermission(user?.role, 'nmap_permissions', 'profiles', 'delete')) {
    actions.push({
      label: t('Delete'),
      value: 'delete',
      onClick: async (selectedRows) => {
        setIds(selectedRows.map((row) => row.id));
        setDeleteDialogOpen(true);
      },
    });
  }

  return (
    <>
      {hasPermission(user?.role, 'nmap_permissions', 'profiles', 'post') && (
        <div className="flex pb-1">
          <AddProfileSheet>
            <Button>
              <Plus />
              {t('Create Profile')}
            </Button>
          </AddProfileSheet>
        </div>
      )}
      {!isLoading && (
        <DataTable
          columns={columns(data || [])}
          data={data || []}
          sorting={[{ id: 'Created At', desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            'Times Ran': false,
            'Created At': false,
          }}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <DeleteProfileDialog ids={ids} open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen} />
    </>
  );
}
