import { ColumnDef } from '@tanstack/react-table';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { Badge } from '@/components/ui/badge';
import { useTranslation } from 'react-i18next';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { GlobeLockIcon, MoreHorizontal } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useState } from 'react';
import SearchSuricata from '@/components/suricata/SearchSuricata';
import TextWithCopyButton from '@/components/ui/TextWithCopyButton';
import SearchRingBufferDialog from '@/components/wireshark/ringbuffer/SearchRingBufferDialog';
import SharkFin from '@/components/ui/icons/SharkFin';
import { useGetIntegrations } from '@/components/queries';

export type HostServiceDetail = {
  hostIp: string;
  port: string;
  protocol: string;
  state: string;
  status: string;
};

function IPAddressCell({ address }: { address: string }) {
  const { t } = useTranslation();
  return address ? <TextWithCopyButton text={address} /> : t('Unknown');
}

function ActionsCell({ details }: { details: HostServiceDetail }) {
  const { t } = useTranslation();
  const [searchSuricataDialogOpen, setSearchSuricataDialogOpen] = useState(false);
  const [searchWiresharkDialogOpen, setSearchWiresharkDialogOpen] = useState(false);
  const { data: integrationStatus } = useGetIntegrations();

  return (
    (!!integrationStatus?.suricata || (!!integrationStatus?.wireshark && !!integrationStatus?.ringbuffer)) && (
      <>
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="ghost" className="h-8 w-8 p-0">
              <span className="sr-only">{t('Open menu')}</span>
              <MoreHorizontal className="h-4 w-4" />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="center">
            {!!integrationStatus?.suricata && (
              <DropdownMenuItem
                onSelect={() => {
                  setSearchSuricataDialogOpen(true);
                }}
              >
                <GlobeLockIcon />
                {t('Search Suricata')}
              </DropdownMenuItem>
            )}
            {!!integrationStatus?.wireshark && integrationStatus?.ringbuffer && (
              <DropdownMenuItem
                onSelect={() => {
                  setSearchWiresharkDialogOpen(true);
                }}
              >
                <SharkFin />
                {t('Search')} Wireshark
              </DropdownMenuItem>
            )}
          </DropdownMenuContent>
        </DropdownMenu>

        <SearchSuricata
          open={searchSuricataDialogOpen}
          onOpenChange={setSearchSuricataDialogOpen}
          srcIp={details.hostIp}
          destIp={details.hostIp}
          srcPort={Number(details.port) || undefined}
          destPort={Number(details.port) || undefined}
          protocol={details.protocol}
        />
        <SearchRingBufferDialog
          open={searchWiresharkDialogOpen}
          onOpenChange={setSearchWiresharkDialogOpen}
          src={details.hostIp}
          dst={details.hostIp}
          srcPort={Number(details.port) || undefined}
          dstPort={Number(details.port) || undefined}
          protocol={details.protocol}
        />
      </>
    )
  );
}

export const hostServiceColumns: ColumnDef<HostServiceDetail>[] = [
  {
    id: 'state',
    accessorKey: 'state',
    header: ({ column }) => <DataTableColumnHeader column={column} title="State" />,
    cell: ({ row }) => (
      <Badge variant={row.original.state === 'open' ? 'default' : 'secondary'}>
        {row.original.state.toUpperCase()}
      </Badge>
    ),
  },
  {
    id: 'IP Address',
    accessorKey: 'hostIp',
    header: ({ column }) => <DataTableColumnHeader column={column} title="IP Address" />,
    cell: ({ row }) => <IPAddressCell address={row.original.hostIp} />,
  },
  {
    accessorKey: 'port',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Port" />,
    cell: ({ row }) => row.original.port,
  },
  {
    accessorKey: 'protocol',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Protocol" />,
    cell: ({ row }) => row.original.protocol,
  },
  {
    id: 'actions',
    cell: ({ row }) => <ActionsCell details={row.original} />,
  },
];
