import { ColumnDef } from '@tanstack/react-table';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { GlobeLockIcon, MoreHorizontal } from 'lucide-react';
import { useState } from 'react';
import SearchSuricata from '@/components/suricata/SearchSuricata';
import { useTranslation } from 'react-i18next';
import SearchRingBufferDialog from '@/components/wireshark/ringbuffer/SearchRingBufferDialog';
import SharkFin from '@/components/ui/icons/SharkFin';
import { useGetIntegrations } from '@/components/queries';

export interface ServiceEntry {
  name: string;
  count: number;
}

function ActionsCell({ protocol }: { protocol: string }) {
  const { t } = useTranslation();
  const [searchSuricataDialogOpen, setSearchSuricataDialogOpen] = useState(false);
  const [searchWiresharkDialogOpen, setSearchWiresharkDialogOpen] = useState(false);
  const { data: integrationStatus } = useGetIntegrations();

  return (
    (!!integrationStatus?.suricata || (!!integrationStatus?.wireshark && !!integrationStatus?.ringbuffer)) && (
      <>
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="ghost" className="h-8 w-8 p-0">
              <span className="sr-only">{t('Open menu')}</span>
              <MoreHorizontal className="h-4 w-4" />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="center">
            {!!integrationStatus?.suricata && (
              <DropdownMenuItem
                onSelect={() => {
                  setSearchSuricataDialogOpen(true);
                }}
              >
                <GlobeLockIcon />
                {t('Search Suricata')}
              </DropdownMenuItem>
            )}
            {!!integrationStatus?.wireshark && integrationStatus?.ringbuffer && (
              <DropdownMenuItem
                onSelect={() => {
                  setSearchWiresharkDialogOpen(true);
                }}
              >
                <SharkFin />
                {t('Search')} Wireshark
              </DropdownMenuItem>
            )}
          </DropdownMenuContent>
        </DropdownMenu>

        <SearchSuricata
          open={searchSuricataDialogOpen}
          onOpenChange={setSearchSuricataDialogOpen}
          protocol={protocol}
        />
        <SearchRingBufferDialog
          open={searchWiresharkDialogOpen}
          onOpenChange={setSearchWiresharkDialogOpen}
          protocol={protocol}
        />
      </>
    )
  );
}

export const serviceColumns: ColumnDef<ServiceEntry>[] = [
  {
    id: 'Service Name',
    accessorKey: 'name',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Service Name" />,
    cell: ({ row }) => <span>{row.original.name}</span>,
  },
  {
    id: 'Host Count',
    accessorKey: 'count',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Host Count" />,
    cell: ({ row }) => <span>{row.original.count}</span>,
  },
  {
    id: 'actions',
    cell: ({ row }) => <ActionsCell protocol={row.original.name} />,
  },
];
