import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import React, { useState } from 'react';
import { useTranslation } from 'react-i18next';
import { useAddScheduledScan, useUpdateScheduledScan } from '@/components/nmap/scheduledscans/queries';
import { scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { nmapProfile } from '@/components/nmap/profiles/types';
import { useGetProfiles } from '@/components/nmap/profiles/queries';
import { PageSeparator } from '@/components/Page';
import { Checkbox } from '@/components/ui/checkbox';
import { validateScanHasTarget, validateScanParameters, validateScanHasInterfaceFlag } from '@/components/nmap/utils';
import { Combobox } from '@/components/ui/combobox';
import { ScheduleFormOptions } from '@/components/scheduling/ScheduleFormOptions';
import { ScheduleParameters } from '@/components/scheduling/types';
import { defaultScheduleParameters } from '@/components/scheduling/utils';

type AddEditScheduledScanDialogProps = {
  scheduledScan?: scheduledNmapScan;
  profile?: nmapProfile;
  setOpen: (open: boolean) => void;
};

function AddEditScheduledScanDialogForm({ scheduledScan, profile, setOpen }: AddEditScheduledScanDialogProps) {
  const [error, setError] = useState<string | null>(null);
  const { t } = useTranslation();
  const addScheduledScanMutation = useAddScheduledScan();
  const editScheduledScanMutation = useUpdateScheduledScan();
  const { data = [] } = useGetProfiles();
  const sortedProfiles = [...data].sort((a, b) => a.name.localeCompare(b.name));

  const form = useForm<scheduledNmapScan>({
    defaultValues: {
      id: scheduledScan?.id ?? undefined,
      name: scheduledScan?.name ?? profile?.name ?? '',
      parameters: scheduledScan?.parameters ?? profile?.parameters ?? '',
      nmap_profile_id: scheduledScan?.nmap_profile_id ?? profile?.id ?? null,
      schedule_type: scheduledScan?.schedule_type ?? 'dailyAt',
      schedule_parameters: scheduledScan ? scheduledScan.schedule_parameters : defaultScheduleParameters,
      enabled: scheduledScan ? !!scheduledScan?.enabled : true,
    },
  });

  const onSubmit: SubmitHandler<scheduledNmapScan> = (data) => {
    if (!validateScanHasTarget(data.parameters || '', setError)) return;
    if (!validateScanParameters(data.parameters || '', setError)) return;
    if (!validateScanHasInterfaceFlag(data.parameters || '', setError)) return;
    if (!scheduledScan) {
      addScheduledScanMutation.mutate(data, {
        onSuccess: () => {
          form.reset();
          setError(null);
          setOpen(false);
        },
      });
    } else {
      const id = scheduledScan.id;
      if (!id) return;

      if (data.schedule_parameters) {
        const cleanedParameters: Record<string, ScheduleParameters[keyof ScheduleParameters]> = {};
        Object.entries(data.schedule_parameters).forEach(([key, value]) => {
          if (Array.isArray(value)) {
            const nonNullValues = value.filter((v) => v !== null && v !== '');
            if (nonNullValues.length > 0) {
              cleanedParameters[key] = value;
            }
          } else {
            if (value !== null && value !== '') {
              cleanedParameters[key] = value;
            }
          }
        });
        data.schedule_parameters = cleanedParameters;
      }

      editScheduledScanMutation.mutate(
        {
          id,
          updates: data,
        },
        {
          onSuccess: () => {
            form.reset();
            setError(null);
            setOpen(false);
          },
        }
      );
    }
  };

  const handleProfileChange = (profileId: string) => {
    const selectedProfile = data.find((p) => String(p.id) === profileId);
    if (selectedProfile) {
      form.setValue('name', selectedProfile.name || '');
      form.setValue('parameters', selectedProfile.parameters || '');
      form.setValue('nmap_profile_id', selectedProfile.id);
      setError(null);
    } else {
      form.setValue('nmap_profile_id', null);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
        {error && <div className="text-destructive mb-2 text-sm">{error}</div>}
        <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
          <FormLabel>{t('Profile')}</FormLabel>
          <div className="col-span-3">
            <Combobox
              options={sortedProfiles.map((p) => ({ value: String(p.id), label: p.name }))}
              value={form.watch('nmap_profile_id') ? String(form.watch('nmap_profile_id')) : ''}
              onChange={handleProfileChange}
              placeholder={t('Select a profile')}
              className="w-full"
            />
          </div>
        </FormItem>
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t('Name')}</FormLabel>
              <FormControl>
                <Input {...field} className="border-border col-span-3" />
              </FormControl>
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="parameters"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t('Scan Parameters')}</FormLabel>
              <FormControl>
                <Input {...field} className="border-border col-span-3" placeholder={t('e.g., -sV -p 80')} />
              </FormControl>
            </FormItem>
          )}
        />
        <PageSeparator />
        <ScheduleFormOptions form={form} />
        <PageSeparator />
        <FormField
          control={form.control}
          name="enabled"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel className="mb-0 cursor-pointer" onClick={() => field.onChange(!field.value)}>
                {t('Enabled')}
              </FormLabel>
              <FormControl>
                <Checkbox checked={field.value} onCheckedChange={field.onChange} />
              </FormControl>
            </FormItem>
          )}
        />
        <DialogFooter>
          <Button type="submit" className="justify-end">
            {t('Submit')}
          </Button>
        </DialogFooter>
      </form>
    </Form>
  );
}

export function AddScheduledScanDialog({ children }: { children: React.ReactNode }) {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>{children}</DialogTrigger>
      <DialogContent className="sm:max-w-[620px]">
        <DialogHeader>
          <DialogTitle>{t('Add Scheduled Scan')}</DialogTitle>
          <DialogDescription>{t('Use this to create a new Scheduled Scan')}</DialogDescription>
        </DialogHeader>
        <AddEditScheduledScanDialogForm setOpen={setOpen} />
      </DialogContent>
    </Dialog>
  );
}

export function EditScheduledScanDialog({
  scheduledScan,
  open,
  onOpenChange,
}: {
  scheduledScan: scheduledNmapScan;
  open: boolean;
  onOpenChange: (open: boolean) => void;
}) {
  const { t } = useTranslation();

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[620px]">
        <DialogHeader>
          <DialogTitle>{t('Edit Scheduled Scan')}</DialogTitle>
          <DialogDescription>{t('Update the details for this Scheduled Scan.')}</DialogDescription>
        </DialogHeader>
        <AddEditScheduledScanDialogForm setOpen={onOpenChange} scheduledScan={scheduledScan} />
      </DialogContent>
    </Dialog>
  );
}

export function AddScheduledScanWithProfileDialog({
  profile,
  open,
  onOpenChange,
}: {
  profile: nmapProfile;
  open: boolean;
  onOpenChange: (open: boolean) => void;
}) {
  const { t } = useTranslation();

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[620px]">
        <DialogHeader>
          <DialogTitle>{t('Add Scheduled Scan')}</DialogTitle>
          <DialogDescription>{t('Use this to create a new Scheduled Scan.')}</DialogDescription>
        </DialogHeader>
        <AddEditScheduledScanDialogForm setOpen={onOpenChange} profile={profile} />
      </DialogContent>
    </Dialog>
  );
}
