import { Button } from '@/components/ui/button';
import { DataTable, DataTableRowAction } from '@/components/ui/DataTable/DataTable';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import React from 'react';
import { columns } from '@/components/nmap/scheduledscans/columns';
import { AddScheduledScanDialog } from '@/components/nmap/scheduledscans/AddEditScheduledScanDialog';
import { DeleteScheduledScanAlertDialog } from '@/components/nmap/scheduledscans/DeleteScheduledScanDialog';
import { scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { Plus } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';

type ScheduledScansProps = {
  data: scheduledNmapScan[] | undefined;
  isLoading: boolean;
};

export default function ScheduledScans({ data, isLoading }: ScheduledScansProps) {
  const { t } = useTranslation();
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [idsToDelete, setIdsToDelete] = React.useState<number[]>([]);
  const { user } = useAuth();

  const actions: DataTableRowAction<scheduledNmapScan>[] = [];

  if (hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'delete')) {
    actions.push({
      label: t('Delete'),
      value: 'delete',
      onClick: async (selectedRows: scheduledNmapScan[]) => {
        setIdsToDelete(selectedRows.map((row: scheduledNmapScan) => row.id));
        setDeleteDialogOpen(true);
      },
    });
  }

  return (
    <>
      {hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'post') && (
        <AddScheduledScanDialog>
          <Button className="mb-1">
            <Plus />
            {t('Schedule a Scan')}
          </Button>
        </AddScheduledScanDialog>
      )}
      {!isLoading && (
        <DataTable
          columns={columns}
          data={data || []}
          sorting={[{ id: 'Last Ran', desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            Duration: false,
            'Finished At': false,
            'Created At': false,
            'Times Ran': false,
          }}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <DeleteScheduledScanAlertDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      />
    </>
  );
}
