import { ColumnDef, Row } from '@tanstack/react-table';
import { Checkbox } from '@/components/ui/checkbox';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { Button } from '@/components/ui/button';
import { useTranslation } from 'react-i18next';
import { MoreHorizontal, Pencil, Trash2 } from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { useState } from 'react';
import { scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { Switch } from '@/components/ui/switch';
import { useUpdateScheduledScan } from '@/components/nmap/scheduledscans/queries';
import { getScheduleTypeLabel } from '@/components/scheduling/utils';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';
import { EditScheduledScanDialog } from '@/components/nmap/scheduledscans/AddEditScheduledScanDialog';
import { DeleteScheduledScanAlertDialog } from '@/components/nmap/scheduledscans/DeleteScheduledScanDialog';
import ScheduleCell from '@/components/scheduling/ScheduleCell';

function CreatedAtCell({ value }: { value: string | null | undefined }) {
  const { t } = useTranslation();

  if (!value || typeof value !== 'string') {
    return <div>{t('Scan Pending')}...</div>;
  }

  const createdAt = new Date(value);
  return <div>{createdAt.toLocaleString()}</div>;
}

function StatusCell({ id, enabled }: { id: number; enabled: boolean }) {
  const { user } = useAuth();
  const updateScheduledScan = useUpdateScheduledScan();
  const [toggled, setToggled] = useState(enabled);

  function handleToggle() {
    setToggled(!toggled);
    updateScheduledScan.mutate(
      { id: id, updates: { enabled: !enabled } },
      {
        onError: () => setToggled(enabled),
      }
    );
  }

  return (
    <Switch
      checked={toggled}
      disabled={!hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'put')}
      onCheckedChange={handleToggle}
    />
  );
}

function ActionsCell({ scheduledScan }: { scheduledScan: scheduledNmapScan }) {
  const { t } = useTranslation();
  const { user } = useAuth();
  const [activeDialog, setActiveDialog] = useState<'edit' | 'delete' | null>(null);

  const canEdit = hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'put');
  const canDelete = hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'delete');
  const canShowMenu = canEdit || canDelete;

  return (
    <>
      {canShowMenu && (
        <>
          <DropdownMenu modal={false}>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" className="h-8 w-8 p-0">
                <span className="sr-only">{t('Open menu')}</span>
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="center">
              {canEdit && (
                <DropdownMenuItem onSelect={() => setActiveDialog('edit')}>
                  <Pencil />
                  {t('Edit')}
                </DropdownMenuItem>
              )}
              {canDelete && (
                <DropdownMenuItem onSelect={() => setActiveDialog('delete')}>
                  <Trash2 />
                  {t('Delete')}
                </DropdownMenuItem>
              )}
            </DropdownMenuContent>
          </DropdownMenu>
          <EditScheduledScanDialog
            scheduledScan={scheduledScan}
            open={activeDialog === 'edit'}
            onOpenChange={(open) => !open && setActiveDialog(null)}
          />
          <DeleteScheduledScanAlertDialog
            ids={[scheduledScan.id]}
            open={activeDialog === 'delete'}
            onOpenChange={(open) => !open && setActiveDialog(null)}
          />
        </>
      )}
    </>
  );
}

export const columns: ColumnDef<scheduledNmapScan>[] = [
  {
    id: 'select',
    header: ({ table }) => (
      <Checkbox
        checked={table.getIsAllPageRowsSelected() || (table.getIsSomePageRowsSelected() && 'indeterminate')}
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
    ),
    cell: ({ row }) => (
      <Checkbox
        checked={row.getIsSelected()}
        onCheckedChange={(value) => row.toggleSelected(!!value)}
        aria-label="Select row"
      />
    ),
    enableSorting: false,
    enableHiding: false,
  },
  {
    accessorKey: 'name',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scheduled Scan Name" />;
    },
  },
  {
    id: 'Scan Parameters',
    accessorKey: 'parameters',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scan Parameters" />;
    },
    cell: ({ row }) => {
      const params = row.getValue('Scan Parameters') as string;
      return <div className="max-w-[200px] break-words whitespace-normal">{params}</div>;
    },
  },
  {
    id: 'Frequency',
    accessorKey: 'schedule_type',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Frequency" />;
    },
    cell: ({ row }) => {
      const type = row.original.schedule_type;
      return <span>{getScheduleTypeLabel(type)}</span>;
    },
    enableSorting: true,
    sortingFn: (rowA, rowB) => {
      const labelA = getScheduleTypeLabel(rowA.original.schedule_type);
      const labelB = getScheduleTypeLabel(rowB.original.schedule_type);
      if (labelA < labelB) return -1;
      if (labelA > labelB) return 1;
      return 0;
    },
  },
  {
    id: 'Schedule Parameters',
    accessorKey: 'schedule_parameters',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Schedule Parameters" />;
    },
    cell: ({ row }) => <ScheduleCell params={row.original.schedule_parameters} type={row.original.schedule_type} />,
    enableSorting: false,
  },
  {
    id: 'Last Ran',
    accessorKey: 'last_run_at',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Last Ran" />;
    },
    cell: ({ row }) => {
      const lastRun = row.original.last_run_at;
      if (!lastRun) return <span>Never</span>;
      const date = new Date(lastRun);
      if (isNaN(date.getTime())) return <span>Invalid date</span>;
      return <span>{date.toLocaleString()}</span>;
    },
    sortingFn: (rowA: Row<scheduledNmapScan>, rowB: Row<scheduledNmapScan>) => {
      function getSortableValue(row: Row<scheduledNmapScan>) {
        const value = row.original.last_run_at;
        if (!value) return -Infinity;
        const date = new Date(value);
        if (isNaN(date.getTime())) return -Infinity;
        return date.getTime();
      }
      const a = getSortableValue(rowA);
      const b = getSortableValue(rowB);
      if (a < b) return -1;
      if (a > b) return 1;
      return 0;
    },
  },
  {
    id: 'Times Ran',
    accessorKey: 'times_ran',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Times Ran" />;
    },
    cell: ({ row }) => {
      return <span>{row.original.times_ran}</span>;
    },
  },
  {
    id: 'Created At',
    accessorKey: 'created_at',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Created At" />;
    },
    cell: ({ row }) => <CreatedAtCell value={row.getValue('Created At')} />,
  },
  {
    id: 'toggle-enabled',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Status" />,
    cell: ({ row }) => <StatusCell id={row.original.id} enabled={row.original.enabled} />,
  },
  {
    id: 'actions',
    cell: ({ row }) => <ActionsCell scheduledScan={row.original} />,
  },
];
