import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import { ColumnDef, Row } from '@tanstack/react-table';
import { Ellipsis, Pencil, Trash2 } from 'lucide-react';
import { Report } from '@/components/report/types';
import { useTranslation } from 'react-i18next';
import { useState } from 'react';
import { Link } from 'react-router-dom';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import { getScheduleTypeLabel } from '@/components/scheduling/utils';
import { Badge } from '../ui/badge';
import { Switch } from '@/components/ui/switch';
import { useDeleteReport, useEditReport } from '@/components/report/queries';
import { HoverCard, HoverCardTrigger, HoverCardContent } from '@/components/ui/hover-card';
import { EditReportDialog } from '@/components/report/AddEditReportDialog';
import { Dialog, DialogTrigger } from '@/components/ui/dialog';
import { useAuth } from '@/contexts/AuthContext';
import { isAdmin } from '@/components/role/utils';
import ScheduleCell from '@/components/scheduling/ScheduleCell';

function StatusCell({ id, enabled }: { id: number; enabled: boolean }) {
  const { user } = useAuth();
  const editReport = useEditReport();
  const [toggled, setToggled] = useState(enabled);

  function handleToggle() {
    setToggled(!toggled);
    editReport.mutate(
      { id: id, updates: { enabled: !enabled } },
      {
        onError: () => setToggled(enabled),
      }
    );
  }

  return <Switch checked={toggled} disabled={!isAdmin(user?.role)} onCheckedChange={handleToggle} />;
}

function ActionsCell({ report }: { report: Report }) {
  const { t } = useTranslation();
  const deleteReportMutation = useDeleteReport();
  const [open, setOpen] = useState(false);

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost">
            <Ellipsis />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent>
          <DialogTrigger asChild>
            <DropdownMenuItem>
              <Pencil />
              {t('Edit')}
            </DropdownMenuItem>
          </DialogTrigger>
          <DropdownMenuItem onClick={() => deleteReportMutation.mutate(report.id)}>
            <Trash2 />
            {t('Delete')}
          </DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
      <EditReportDialog report={report} setOpen={setOpen} />
    </Dialog>
  );
}

type ReportsDataTableProps = {
  reports: Report[];
  isLoading?: boolean;
};

export default function ReportsDataTable({ reports, isLoading }: ReportsDataTableProps) {
  const { user } = useAuth();
  const { t } = useTranslation();

  const columns: ColumnDef<Report>[] = [
    {
      accessorKey: 'name',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t('Report Name')} />,
      cell: ({ row }) => (
        <Link className="text-primary" to={`/dashboards/${row.original.dashboard_id}`}>
          {row.original.name}
        </Link>
      ),
    },
    {
      accessorKey: 'schedule',
      header: t('Schedule'),
      cell: ({ row }) => getScheduleTypeLabel(row.original.schedule_type),
    },
    {
      id: 'Schedule Parameters',
      accessorKey: 'schedule_parameters',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Schedule Parameters" />;
      },
      cell: ({ row }) => <ScheduleCell params={row.original.schedule_parameters} type={row.original.schedule_type} />,
      enableSorting: false,
    },
    {
      accessorKey: 'format',
      header: t('Format'),
      cell: ({ row }) => row.original.format.toUpperCase(),
    },
    {
      accessorKey: 'recipients',
      header: t('Recipients'),
      cell: ({ row }) => (
        <div className="flex items-center gap-1">
          {row.original.users.map((user, index, arr) => {
            if (index < 3) return <Badge key={user.id} variant="outline">{`${user.username} (${user.email})`}</Badge>;
            else if (index === 3)
              return (
                <HoverCard openDelay={50} closeDelay={50} key={user.id}>
                  <HoverCardTrigger>
                    <Badge variant="outline" className="hover:cursor-default">
                      + {arr.length - 3}
                    </Badge>
                  </HoverCardTrigger>
                  <HoverCardContent
                    side="right"
                    className="flex w-fit flex-col space-y-1 border-none bg-transparent p-0"
                  >
                    {arr.slice(3).map((user) => (
                      <Badge
                        key={user.id}
                        variant="outline"
                        className="bg-background"
                      >{`${user.username} (${user.email})`}</Badge>
                    ))}
                  </HoverCardContent>
                </HoverCard>
              );
          })}
        </div>
      ),
    },
    {
      accessorKey: 'enabled',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t('Status')} />,
      cell: ({ row }) => <StatusCell id={row.original.id} enabled={row.original.enabled} />,
    },
    ...(isAdmin(user?.role)
      ? [
          {
            id: 'actions',
            header: '',
            cell: ({ row }: { row: Row<Report> }) => <ActionsCell report={row.original} />,
          },
        ]
      : []),
  ];

  return <>{isLoading ? <DataTableSkeleton></DataTableSkeleton> : <DataTable columns={columns} data={reports} />}</>;
}
