import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { Report } from '@/components/report/types';
import { toast } from 'sonner';
import { useTranslation } from 'react-i18next';
import { useNavigate } from 'react-router-dom';

export function useGetReports() {
  return useQuery<Report[], AxiosError>({
    queryKey: ['reports'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/reports');
      return response.data;
    },
  });
}

export function useAddReport() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const navigate = useNavigate();
  return useMutation({
    mutationFn: async (report: Report & { source: number }) => {
      const response = await axios.post(`/api/v1/reports`, report);
      return response.data;
    },
    onSuccess: (_response, variables) => {
      queryClient.invalidateQueries({ queryKey: ['reports'] });
      toast(t('Report created.'), {
        ...(variables.source !== -1 && {
          action: {
            label: t('View'),
            onClick: () => navigate('/reports'),
          },
        }),
      });
    },
    onError: () => {
      toast(t('Failed to create report.'), { description: t('Please try again.') });
    },
  });
}

export function useEditReport() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async ({ id, updates }: { id: number; updates: Partial<Report> }) => {
      const response = await axios.patch(`/api/v1/reports/${id}`, updates);
      return response.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['reports'] });
      toast(t('Report edited.'));
    },
    onError: () => {
      toast.error(t('Failed to edit report.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteReport() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (id: number) => {
      const response = await axios.delete(`/api/v1/reports/${id}`);
      return response.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['reports'] });
      toast(t('Report deleted'));
    },
    onError: () => {
      toast(t('Failed to delete report'), { description: t('Please try again') });
    },
  });
}
