import { Crud, Role } from '@/components/role/types';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { Form, FormLabel, FormControl, FormField, FormItem, FormDescription, FormMessage } from '@/components/ui/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import React, { useRef, useState } from 'react';
import { useTranslation } from 'react-i18next';
import { useAddRole, useEditRole, useGetRole } from '@/components/role/queries';
import { GlobeLockIcon, Network, Radar, SquareUser } from 'lucide-react';
import SharkFin from '@/components/ui/icons/SharkFin';
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarProvider,
} from '@/components/ui/sidebar';
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from '@/components/ui/breadcrumb';
import { Card, CardContent } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Switch } from '@/components/ui/switch';
import { Label } from '@/components/ui/label';
import { PageSeparator } from '@/components/Page';
import { Input } from '../ui/input';
import { allDefault, noneDefault } from '@/components/role/defaults';
import { RadioGroup, RadioGroupItem } from '../ui/radio-group';
import { ScrollingContainer } from '@/components/ui/scrolling-container';

type AddEditRoleDialogProps = {
  role?: Role;
  setOpen: (open: boolean) => void;
  userRole: Role;
};

function AddEditRoleDialog({ role, setOpen, userRole }: AddEditRoleDialogProps) {
  const { t } = useTranslation();
  const addRoleMutation = useAddRole();
  const editRoleMutation = useEditRole();
  const sectionRefs = useRef<Record<string, HTMLDivElement | null>>({});
  const [activeSection, setActiveSection] = useState('role-info');
  const containerRef = useRef<HTMLDivElement | null>(null);

  const form = useForm<Role>({
    defaultValues: {
      id: role ? role.id : -1,
      name: role ? role.name : t('New ') + userRole.name,
      protected: role ? role.protected : userRole.protected,
      flow_source_permissions: role ? role.flow_source_permissions : userRole.flow_source_permissions,
      suricata_permissions: role ? role.suricata_permissions : userRole.suricata_permissions,
      wireshark_permissions: role ? role.wireshark_permissions : userRole.wireshark_permissions,
      nmap_permissions: role ? role.nmap_permissions : userRole.nmap_permissions,
    },
  });

  const nav = [
    { name: t('Role Information'), icon: SquareUser, fragment: 'role-info' },
    { name: t('Flow Sources'), icon: Network, fragment: 'flow' },
    { name: 'Wireshark', icon: SharkFin, fragment: 'wireshark' },
    { name: 'Nmap', icon: Radar, fragment: 'nmap' },
    { name: 'Suricata', icon: GlobeLockIcon, fragment: 'suricata' },
  ];

  const crud: { id: Crud; label: string }[] = [
    { id: 'get', label: t('View') },
    { id: 'post', label: t('Create') },
    { id: 'put', label: t('Edit') },
    { id: 'delete', label: t('Delete') },
  ];

  const onSubmit: SubmitHandler<Role> = (data) => {
    if (role) {
      editRoleMutation.mutate(data);
    } else {
      addRoleMutation.mutate(data);
    }
    setOpen(false);
  };

  return (
    <SidebarProvider className="items-start">
      <Sidebar collapsible="none" className="hidden lg:flex">
        <SidebarContent className="border-r">
          <SidebarGroup>
            <SidebarGroupContent>
              <SidebarMenu>
                {nav.map((item) => (
                  <SidebarMenuItem key={item.name}>
                    <SidebarMenuButton
                      asChild
                      isActive={activeSection === item.fragment}
                      className="hover:cursor-pointer"
                    >
                      <a
                        onClick={() => {
                          setActiveSection(item.fragment);
                          const target = sectionRefs.current[item.fragment];
                          const container = containerRef?.current?.querySelector('[data-radix-scroll-area-viewport]');
                          if (target && container) {
                            const containerTop = container.getBoundingClientRect().top;
                            const childTop = target.getBoundingClientRect().top;
                            const offset = childTop - containerTop;

                            container.scrollTo({
                              top: offset + container.scrollTop,
                            });
                          }
                        }}
                      >
                        <item.icon />
                        <span>{item.name}</span>
                      </a>
                    </SidebarMenuButton>
                  </SidebarMenuItem>
                ))}
              </SidebarMenu>
            </SidebarGroupContent>
          </SidebarGroup>
        </SidebarContent>
      </Sidebar>
      <div className="flex flex-1 flex-col">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <div className="flex h-16 items-center gap-2 px-4">
              <Breadcrumb>
                <BreadcrumbList>
                  <BreadcrumbItem className="hidden md:block">{t('Roles')}</BreadcrumbItem>
                  <BreadcrumbSeparator className="hidden md:block" />
                  <BreadcrumbItem>
                    <BreadcrumbPage>{role ? t('Edit') : t('Create')}</BreadcrumbPage>
                  </BreadcrumbItem>
                </BreadcrumbList>
              </Breadcrumb>
            </div>
            <ScrollingContainer
              threshold={0.2}
              setActiveSection={setActiveSection}
              sectionRefs={sectionRefs}
              containerRef={containerRef}
            >
              <Card
                ref={(el) => {
                  sectionRefs.current['role-info'] = el;
                }}
                id="role-info"
              >
                <CardContent>
                  <div className="mb-6 flex items-center gap-2 text-lg font-semibold">
                    <SquareUser />
                    {t('Role Information')}
                  </div>
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Name')}</FormLabel>
                          <FormDescription>{t('A unique reference to this role')}</FormDescription>
                        </div>
                        <FormControl>
                          <Input {...field} className="w-sm" disabled={role?.protected} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <div className="mb-4 flex items-center justify-between gap-4">
                    <div className="space-y-1">
                      <FormLabel>{t('Presets')}</FormLabel>
                      <FormDescription>{t('Optionally select from a default permission preset')}</FormDescription>
                    </div>
                    <RadioGroup
                      className="grid-cols-3"
                      defaultValue={role ? '' : 'user'}
                      onValueChange={(value) => {
                        const newId = role ? role.id : -1;
                        const newName = form.getValues('name');
                        const newType = role ? role.type : 'custom';
                        switch (value) {
                          case 'all':
                            form.reset({ ...allDefault, id: newId, name: newName, type: newType });
                            break;
                          case 'none':
                            form.reset({ ...noneDefault, id: newId, name: newName, type: newType });
                            break;
                          case 'user':
                            form.reset({ ...userRole, id: newId, name: newName, type: newType });
                            break;
                          default:
                            break;
                        }
                      }}
                    >
                      <div className="flex flex-col items-center gap-2">
                        <Label htmlFor="all">{t(`All`)}</Label>
                        <RadioGroupItem value="all" id="all" />
                      </div>
                      <div className="flex flex-col items-center gap-2">
                        <Label htmlFor="none">{t(`None`)}</Label>
                        <RadioGroupItem value="none" id="none" />
                      </div>
                      <div className="flex flex-col items-center gap-2">
                        <Label htmlFor="user">{t(`User`)}</Label>
                        <RadioGroupItem value="user" id="user" />
                      </div>
                    </RadioGroup>
                  </div>
                </CardContent>
              </Card>
              <Card
                ref={(el) => {
                  sectionRefs.current['flow'] = el;
                }}
                id="flow"
              >
                <CardContent>
                  <div className="mb-6 flex items-center gap-2 text-lg font-semibold">
                    <Network />
                    {t('Flow Sources')}
                  </div>
                  <FormField
                    control={form.control}
                    name="flow_source_permissions.sources"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Sources')}</FormLabel>
                          <FormDescription>{t('Control access to flow sources')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="flow_source_permissions.sources"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="flow_source_permissions.sourcegroups"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Groups')}</FormLabel>
                          <FormDescription>{t('Control access to flow source groups')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="flow_source_permissions.sourcegroups"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="flow_source_permissions.start_stop_sources"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Start & Stop Sources')}</FormLabel>
                          <FormDescription>{t('Allow this user to start & stop sources')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="flow_source_permissions.start_stop_sourcegroups"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Start & Stop Sourcegroups')}</FormLabel>
                          <FormDescription>{t('Allow this user to start & stop source groups')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>
              <Card
                ref={(el) => {
                  sectionRefs.current['wireshark'] = el;
                }}
                id="wireshark"
              >
                <CardContent>
                  <div className="mb-6 flex items-center gap-2 text-lg font-semibold">
                    <SharkFin />
                    Wireshark
                  </div>
                  <FormField
                    control={form.control}
                    name="wireshark_permissions.pcaps"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('PCAP Files')}</FormLabel>
                          <FormDescription>{t('Control access to Wireshark PCAP files')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div />
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length >= 3}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud
                            .filter((op) => op.id !== 'put')
                            .map((op) => (
                              <FormField
                                key={op.id}
                                control={form.control}
                                name="wireshark_permissions.pcaps"
                                render={({ field }) => {
                                  return (
                                    <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                      <FormLabel className="text-sm font-normal">
                                        {op.id === 'post' ? t('Import') : op.label}
                                      </FormLabel>
                                      <FormControl>
                                        <Checkbox
                                          checked={field.value?.includes(op.id)}
                                          onCheckedChange={(checked) => {
                                            return checked
                                              ? field.onChange([...field.value, op.id])
                                              : field.onChange(field.value?.filter((value) => value !== op.id));
                                          }}
                                        />
                                      </FormControl>
                                    </FormItem>
                                  );
                                }}
                              />
                            ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="wireshark_permissions.start_stop_capture"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Start & Stop Capture')}</FormLabel>
                          <FormDescription>{t('Allow this user to start & stop traffic capturing')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="wireshark_permissions.ring_buffer"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Ring Buffer')}</FormLabel>
                          <FormDescription>{t('Control access to the Wireshark ring buffer')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div />
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length >= 3}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud
                            .filter((op) => op.id !== 'delete')
                            .map((op) => (
                              <FormField
                                key={op.id}
                                control={form.control}
                                name="wireshark_permissions.ring_buffer"
                                render={({ field }) => {
                                  return (
                                    <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                      <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                      <FormControl>
                                        <Checkbox
                                          checked={field.value?.includes(op.id)}
                                          onCheckedChange={(checked) => {
                                            return checked
                                              ? field.onChange([...field.value, op.id])
                                              : field.onChange(field.value?.filter((value) => value !== op.id));
                                          }}
                                        />
                                      </FormControl>
                                    </FormItem>
                                  );
                                }}
                              />
                            ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="wireshark_permissions.start_stop_ring_buffer"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Start & Stop Ring Buffer')}</FormLabel>
                          <FormDescription>{t('Allow this user to start & stop the ring buffer')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>
              <Card
                ref={(el) => {
                  sectionRefs.current['nmap'] = el;
                }}
                id="nmap"
              >
                <CardContent>
                  <div className="mb-6 flex items-center gap-2 text-lg font-semibold">
                    <Radar />
                    Nmap
                  </div>
                  <FormField
                    control={form.control}
                    name="nmap_permissions.scans"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Scans')}</FormLabel>
                          <FormDescription>{t('Control access to Nmap scans')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="nmap_permissions.scans"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">
                                      {op.id === 'put' ? t('Stop') : op.label}
                                    </FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="nmap_permissions.scheduled_scans"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Scheduled Scans')}</FormLabel>
                          <FormDescription>{t('Control access to scheduled nmap scans')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="nmap_permissions.scheduled_scans"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="nmap_permissions.profiles"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Profiles')}</FormLabel>
                          <FormDescription>{t('Control access to nmap profiles')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="nmap_permissions.profiles"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="nmap_permissions.ndiffs"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Ndiffs')}</FormLabel>
                          <FormDescription>{t('Control access to ndiffs')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div />
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length >= 3}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud
                            .filter((op) => op.id !== 'put')
                            .map((op) => (
                              <FormField
                                key={op.id}
                                control={form.control}
                                name="nmap_permissions.ndiffs"
                                render={({ field }) => {
                                  return (
                                    <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                      <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                      <FormControl>
                                        <Checkbox
                                          checked={field.value?.includes(op.id)}
                                          onCheckedChange={(checked) => {
                                            return checked
                                              ? field.onChange([...field.value, op.id])
                                              : field.onChange(field.value?.filter((value) => value !== op.id));
                                          }}
                                        />
                                      </FormControl>
                                    </FormItem>
                                  );
                                }}
                              />
                            ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>
              <Card
                ref={(el) => {
                  sectionRefs.current['suricata'] = el;
                }}
                id="suricata"
              >
                <CardContent>
                  <div className="mb-6 flex items-center gap-2 text-lg font-semibold">
                    <GlobeLockIcon />
                    Suricata
                  </div>
                  <FormField
                    control={form.control}
                    name="suricata_permissions.rules"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Rules')}</FormLabel>
                          <FormDescription>{t('Control access to Suricata rules')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="suricata_permissions.rules"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.rulesets"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Rulesets')}</FormLabel>
                          <FormDescription>{t('Control access to Suricata rulesets')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length === 4}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'post', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud.map((op) => (
                            <FormField
                              key={op.id}
                              control={form.control}
                              name="suricata_permissions.rulesets"
                              render={({ field }) => {
                                return (
                                  <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                    <FormLabel className="text-sm font-normal">{op.label}</FormLabel>
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.includes(op.id)}
                                        onCheckedChange={(checked) => {
                                          return checked
                                            ? field.onChange([...field.value, op.id])
                                            : field.onChange(field.value?.filter((value) => value !== op.id));
                                        }}
                                      />
                                    </FormControl>
                                  </FormItem>
                                );
                              }}
                            />
                          ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.data"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Data')}</FormLabel>
                          <FormDescription>{t('Control access to Suricata data')}</FormDescription>
                        </div>
                        <div className="grid w-60 grid-cols-5 gap-4">
                          <div />
                          <div className="flex flex-col items-center gap-2">
                            <Label className="text-sm font-normal">{t('All')}</Label>
                            <Checkbox
                              checked={field.value.length >= 3}
                              onCheckedChange={(checked) => {
                                return checked ? field.onChange(['get', 'put', 'delete']) : field.onChange([]);
                              }}
                            />
                          </div>
                          {crud
                            .filter((op) => op.id !== 'post')
                            .map((op) => (
                              <FormField
                                key={op.id}
                                control={form.control}
                                name="suricata_permissions.data"
                                render={({ field }) => {
                                  return (
                                    <FormItem key={op.id} className="flex flex-col items-center gap-2">
                                      <FormLabel className="text-sm font-normal">
                                        {op.id === 'put' ? t('Actions') : op.label}
                                      </FormLabel>
                                      <FormControl>
                                        <Checkbox
                                          checked={field.value?.includes(op.id)}
                                          onCheckedChange={(checked) => {
                                            return checked
                                              ? field.onChange([...field.value, op.id])
                                              : field.onChange(field.value?.filter((value) => value !== op.id));
                                          }}
                                        />
                                      </FormControl>
                                    </FormItem>
                                  );
                                }}
                              />
                            ))}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.config"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Config')}</FormLabel>
                          <FormDescription>{t('Allow this user to edit the Suricata configuration')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.alerts"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Alerts')}</FormLabel>
                          <FormDescription>{t('Allow this user to view Suricata alerts')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.start_stop_scan"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Start & Stop Scan')}</FormLabel>
                          <FormDescription>{t('Allow this user to start & stop Suricata scans')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <PageSeparator />
                  <FormField
                    control={form.control}
                    name="suricata_permissions.scan_pcap"
                    render={({ field }) => (
                      <FormItem className="mb-4 flex flex-wrap items-center justify-between gap-4">
                        <div className="space-y-1">
                          <FormLabel>{t('Scan PCAPs')}</FormLabel>
                          <FormDescription>{t('Allow this user to Scan PCAP files with Suricata')}</FormDescription>
                        </div>
                        <FormControl>
                          <Switch checked={field.value} onCheckedChange={field.onChange} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>
            </ScrollingContainer>
            <DialogFooter className="h-16 p-3.5">
              <Button type="button" variant="secondary" onClick={() => setOpen(false)}>
                {t('Cancel')}
              </Button>
              <Button type="submit">{t('Submit')}</Button>
            </DialogFooter>
          </form>
        </Form>
      </div>
    </SidebarProvider>
  );
}
export function AddRoleDialog({ children }: { children: React.ReactNode }) {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();
  const { data: userRole } = useGetRole('user');

  if (!userRole) return children;

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>{children}</DialogTrigger>
      <DialogContent className="overflow-hidden p-0 sm:max-h-[700px] sm:max-w-[700px] lg:max-w-[1100px]">
        <DialogTitle className="sr-only">{t('Create Role')}</DialogTitle>
        <DialogDescription className="sr-only">{t('Use this to create a new role')}</DialogDescription>
        <AddEditRoleDialog setOpen={setOpen} userRole={userRole} />
      </DialogContent>
    </Dialog>
  );
}

export function EditRoleDialog({ role, setOpen }: { role: Role; setOpen: (open: boolean) => void }) {
  const { t } = useTranslation();
  const { data: userRole } = useGetRole('user');

  if (!userRole) return;

  return (
    <DialogContent className="overflow-hidden p-0 sm:max-h-[700px] sm:max-w-[700px] lg:max-w-[1100px]">
      <DialogTitle className="sr-only">{t('Edit Role')}</DialogTitle>
      <DialogDescription className="sr-only">{t('Use this to edit an existing role.')}</DialogDescription>
      <AddEditRoleDialog setOpen={setOpen} role={role} userRole={userRole} />
    </DialogContent>
  );
}
