import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { Role } from '@/components/role/types';
import { toast } from 'sonner';
import { useTranslation } from 'react-i18next';

export function useGetRoles() {
  return useQuery<Role[], AxiosError>({
    queryKey: ['roles'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/roles');
      return response.data;
    },
  });
}

export function useGetRole(type: string) {
  return useQuery<Role, AxiosError>({
    queryKey: ['role'],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/roles/${type}`);
      return response.data;
    },
  });
}

export function useAddRole() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (role: Role) => {
      const response = await axios.post(`/api/v1/roles`, role);
      return response.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['roles'] });
      toast(t('Role created.'));
    },
    onError: () => {
      toast(t('Failed to create role.'), { description: t('Please try again.') });
    },
  });
}

export function useEditRole() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (role: Role) => {
      const response = await axios.put(`/api/v1/roles/${role.id}`, role);
      return response.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['roles'] });
      toast(t('Role edited.'));
    },
    onError: () => {
      toast.error(t('Failed to edit role.'), { description: t('Please try again.') });
    },
  });
}

export function useDeleteRole() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (id: number) => {
      const response = await axios.delete(`/api/v1/roles/${id}`);
      return response.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['roles'] });
      toast(t('Role deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete role.'), { description: t('Please try again') });
    },
  });
}
