import { useTranslation } from 'react-i18next';
import { getDayOfMonthWithSuffix } from '@/components/scheduling/utils';
import { ScheduleType, ScheduleParameters } from '@/components/scheduling/types';

function formatTime(timeStr: string) {
  if (!timeStr) return '';
  const [hour, minute] = timeStr.split(':');
  const h = typeof hour === 'string' ? parseInt(hour, 10) : hour;
  const m = typeof minute === 'string' ? parseInt(minute, 10) : minute;
  if (isNaN(h)) return hour;
  const suffix = h >= 12 ? 'PM' : 'AM';
  let hour12 = h % 12;
  if (hour12 === 0) hour12 = 12;
  const minStr = typeof m === 'number' && !isNaN(m) ? `:${m.toString().padStart(2, '0')}` : '';
  return `${hour12}${minStr} ${suffix}`;
}

export default function ScheduleCell({ params, type }: { params: ScheduleParameters; type: ScheduleType }) {
  const { t } = useTranslation();

  if (!params || Object.keys(params).length === 0) return <span>-</span>;

  const dayNames = [t('Sunday'), t('Monday'), t('Tuesday'), t('Wednesday'), t('Thursday'), t('Friday'), t('Saturday')];
  const monthNames = [
    t('January'),
    t('February'),
    t('March'),
    t('April'),
    t('May'),
    t('June'),
    t('July'),
    t('August'),
    t('September'),
    t('October'),
    t('November'),
    t('December'),
  ];

  return (
    <div className="max-w-[300px] break-words whitespace-pre-line">
      {type === 'cron' && params.cron && (
        <div className="max-w-50 truncate">
          <strong>{t('Expression')}: </strong>
          <span>{params.cron}</span>
          <br />
        </div>
      )}

      {(type === 'every' || type === 'hourlyAt') && params.days && (
        <>
          <strong>{t('Days of Week')}: </strong>
          {params.days
            .sort()
            .map((day: string | number) => dayNames[Number(day)] || day)
            .join(', ')}
          <br />
        </>
      )}

      {type === 'weeklyOn' && params.weekday && (
        <>
          <strong>{t('Day of Week')}: </strong>
          {dayNames[Number(params.weekday)]}
          <br />
        </>
      )}

      {type === 'yearlyOn' && params.month && (
        <>
          <strong>{t('Month')}: </strong>
          {monthNames[Number(params.month) - 1] || params.month}
          <br />
        </>
      )}

      {(type === 'monthlyOn' || type === 'yearlyOn') && params.day && (
        <>
          <strong>{t('Day')}: </strong>
          {getDayOfMonthWithSuffix(params.day)}
          <br />
        </>
      )}

      {(type === 'every' || type === 'hourlyAt') && params.between && (
        <>
          <strong>{t('Between')}: </strong>
          {formatTime(params.between[0])}
          <span> {t('and')} </span>
          {formatTime(params.between[1])}
          <br />
        </>
      )}

      {(type === 'dailyAt' || type === 'weeklyOn' || type === 'monthlyOn' || type === 'yearlyOn') && params.time && (
        <>
          <strong>{t('Time')}: </strong>
          {formatTime(params.time)}
          <br />
        </>
      )}

      {type === 'every' && params.x && (
        <>
          <strong>{t('X')}: </strong>
          {params.x}
          <br />
        </>
      )}

      {type === 'hourlyAt' && params.minute && (
        <>
          <strong>{t('Minute')}: </strong>
          {params.minute}
          <br />
        </>
      )}
    </div>
  );
}
