import { FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form';
import { Select, SelectContent, SelectGroup, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useTranslation } from 'react-i18next';
import { getDayOfMonthWithSuffix, defaultScheduleParameters } from '@/components/scheduling/utils';
import { MultiSelect } from '@/components/ui/multi-select';
import { Input } from '@/components/ui/input';
import { FieldPath, PathValue, UseFormReturn } from 'react-hook-form';
import { Button } from '@/components/ui/button';
import { Calendar } from '@/components/ui/calendar';
import * as React from 'react';
import { Label } from '@/components/ui/label';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { CalendarIcon } from 'lucide-react';
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from '@/components/ui/accordion';
import TimePicker from '@/components/ui/TimePicker';
import { cn } from '@/lib/utils';
import { Schedule } from '@/components/scheduling/types';

const BASE_YEAR = new Date().getFullYear();

export function ScheduleFormOptions<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();
  const selectedPreset = form.watch('schedule_type' as FieldPath<T>);

  const presets = [
    { label: 'Every', value: 'every' },
    { label: 'Hourly', value: 'hourlyAt' },
    { label: 'Daily', value: 'dailyAt' },
    { label: 'Weekly', value: 'weeklyOn' },
    { label: 'Monthly', value: 'monthlyOn' },
    { label: 'Yearly', value: 'yearlyOn' },
    { label: 'Custom Cron', value: 'cron' },
  ];

  const presetFields: Record<string, React.ReactNode> = {
    every: <XField form={form} />,
    hourlyAt: <MinuteField form={form} />,
    dailyAt: <TimeField form={form} />,
    weeklyOn: (
      <>
        <WeekdayField form={form} />
        <TimeField form={form} />
      </>
    ),
    monthlyOn: (
      <>
        <MonthField form={form} />
        <TimeField form={form} />
      </>
    ),
    yearlyOn: (
      <>
        <DateField form={form} />
        <TimeField form={form} />
      </>
    ),
    cron: <CronField form={form} />,
  };

  return (
    <Accordion type="single" collapsible className="w-full" defaultValue="">
      <AccordionItem value="advanced">
        <FormField
          control={form.control}
          name={'schedule_type' as FieldPath<T>}
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t('Schedule')}</FormLabel>
              <FormControl>
                <Select
                  value={field.value as string}
                  onValueChange={(value) => {
                    field.onChange(value);
                    form.resetField('schedule_parameters' as FieldPath<T>, {
                      defaultValue: defaultScheduleParameters as PathValue<T, FieldPath<T>>,
                    });
                  }}
                >
                  <SelectTrigger className="col-span-2 w-full">
                    <SelectValue placeholder={t('Select a schedule')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectGroup>
                      {presets.map(({ label, value }) => (
                        <SelectItem key={value} value={value}>
                          {label}
                          {value === 'every' && ' ' + t('Minute')}
                        </SelectItem>
                      ))}
                    </SelectGroup>
                  </SelectContent>
                </Select>
              </FormControl>
              <AccordionTrigger className="m-0 justify-end p-0">{t('Advanced')}</AccordionTrigger>
            </FormItem>
          )}
        />
        <AccordionContent
          className={cn(
            'bg-background mt-4 rounded-lg border px-6 py-4 transition-[height] duration-200',
            selectedPreset === 'every' || selectedPreset === 'hourlyAt' ? 'h-48 divide-y' : 'h-18'
          )}
        >
          <div className="grid grid-cols-4 items-center gap-2 pb-4">
            <Label>{presets.find((preset) => preset.value === selectedPreset)?.label}</Label>
            <div className="col-span-3 flex gap-2">{presetFields[selectedPreset]}</div>
          </div>
          <div className="grid grid-rows-2 gap-4 pt-4">
            <DaysField form={form} />
            <BetweenField form={form} />
          </div>
        </AccordionContent>
      </AccordionItem>
    </Accordion>
  );
}

function XField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();
  const minutes = [1, 5, 10, 15, 30];

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.x` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('X')}</FormLabel>
          <FormControl>
            <Select value={field.value as string} onValueChange={field.onChange}>
              <SelectTrigger>
                <SelectValue placeholder="X" />
              </SelectTrigger>
              <SelectContent>
                <SelectGroup>
                  {minutes.map((minute) => (
                    <SelectItem key={minute} value={`${minute}`}>
                      {minute}
                    </SelectItem>
                  ))}
                </SelectGroup>
              </SelectContent>
            </Select>
          </FormControl>
          <span>{t('minutes')}</span>
        </FormItem>
      )}
    />
  );
}

function MinuteField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.minute` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('Minute')}</FormLabel>
          <span>{t('at')}</span>
          <FormControl>
            <Select value={field.value as string} onValueChange={field.onChange}>
              <SelectTrigger>
                <SelectValue placeholder="X" />
              </SelectTrigger>
              <SelectContent>
                <SelectGroup>
                  {Array.from({ length: 6 }, (_, i) => i * 10).map((minute) => {
                    const display = minute.toString().padStart(2, '0');
                    return (
                      <SelectItem key={minute} value={`${minute}`}>
                        {display}
                      </SelectItem>
                    );
                  })}
                </SelectGroup>
              </SelectContent>
            </Select>
          </FormControl>
          <span>{t('minutes')}</span>
        </FormItem>
      )}
    />
  );
}

function TimeField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.time` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('Time')}</FormLabel>
          <span>{t('at')}</span>
          <FormControl>
            <TimePicker
              mode="default"
              value={(field.value as string[]) || ['', '']}
              onChange={field.onChange}
              className="col-span-3 w-full justify-between"
            />
          </FormControl>
        </FormItem>
      )}
    />
  );
}

function WeekdayField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();
  const days = [
    { value: '0', label: t('Sunday') },
    { value: '1', label: t('Monday') },
    { value: '2', label: t('Tuesday') },
    { value: '3', label: t('Wednesday') },
    { value: '4', label: t('Thursday') },
    { value: '5', label: t('Friday') },
    { value: '6', label: t('Saturday') },
  ];

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.weekday` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('Weekday')}</FormLabel>
          <span>{t('on')}</span>
          <FormControl>
            <Select value={field.value as string} onValueChange={field.onChange}>
              <SelectTrigger>
                <SelectValue placeholder="day" />
              </SelectTrigger>
              <SelectContent>
                <SelectGroup>
                  {days.map(({ label, value }) => (
                    <SelectItem key={value} value={value}>
                      {label}
                    </SelectItem>
                  ))}
                </SelectGroup>
              </SelectContent>
            </Select>
          </FormControl>
        </FormItem>
      )}
    />
  );
}

function MonthField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.month` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('Month')}</FormLabel>
          <span>{t('on')}</span>
          <FormControl>
            <Select value={field.value as string} onValueChange={field.onChange}>
              <SelectTrigger>
                <SelectValue placeholder="day" />
              </SelectTrigger>
              <SelectContent className="max-h-60">
                <SelectGroup>
                  {Array.from({ length: 31 }, (_, i) => i + 1).map((day) => (
                    <SelectItem key={day} value={`${day}`}>
                      {getDayOfMonthWithSuffix(day)}
                    </SelectItem>
                  ))}
                </SelectGroup>
              </SelectContent>
            </Select>
          </FormControl>
        </FormItem>
      )}
    />
  );
}

function DateField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();
  const [open, setOpen] = React.useState(false);
  const [visibleMonth, setVisibleMonth] = React.useState(new Date(BASE_YEAR, 0));

  const month = form.watch('schedule_parameters.month' as FieldPath<T>);
  const day = form.watch('schedule_parameters.day' as FieldPath<T>);

  const selectedDate = month != null && day != null ? new Date(BASE_YEAR, Number(month) - 1, Number(day)) : undefined;

  function handleMonthChange(nextMonth: Date) {
    const next = nextMonth.getMonth();
    const current = visibleMonth.getMonth();

    let newMonth = next;
    if (current === 11 && next === 0) {
      newMonth = 0;
    } else if (current === 0 && next === 11) {
      newMonth = 11;
    }

    setVisibleMonth(new Date(BASE_YEAR, newMonth));
  }

  return (
    <FormItem className="flex items-center gap-2 whitespace-nowrap">
      <FormLabel className="sr-only">{t('Date')}</FormLabel>
      <span>{t('on')}</span>
      <FormControl>
        <Popover open={open} onOpenChange={setOpen}>
          <PopoverTrigger asChild>
            <Button
              variant="outline"
              id="date-picker"
              className="w-32 justify-between bg-transparent font-normal dark:bg-transparent"
            >
              {month && day
                ? `${new Date(BASE_YEAR, Number(month) - 1, Number(day)).toLocaleString('default', {
                    month: 'long',
                  })} ${day}`
                : 'Select date'}
              <CalendarIcon />
            </Button>
          </PopoverTrigger>
          <PopoverContent className="w-auto overflow-hidden p-0" align="start">
            <Calendar
              mode="single"
              month={visibleMonth}
              onMonthChange={handleMonthChange}
              selected={selectedDate}
              formatters={{
                formatCaption: (date: Date) => date.toLocaleString('default', { month: 'long' }),
              }}
              onSelect={(date: Date | undefined) => {
                if (!date) return;
                const normalized = new Date(BASE_YEAR, date.getMonth(), date.getDate());
                form.setValue(
                  'schedule_parameters.month' as FieldPath<T>,
                  `${normalized.getMonth() + 1}` as PathValue<T, FieldPath<T>>
                );
                form.setValue(
                  'schedule_parameters.day' as FieldPath<T>,
                  `${normalized.getDate()}` as PathValue<T, FieldPath<T>>
                );
              }}
            />
          </PopoverContent>
        </Popover>
      </FormControl>
    </FormItem>
  );
}

function CronField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.cron` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="flex items-center gap-2 whitespace-nowrap">
          <FormLabel className="sr-only">{t('Cron')}</FormLabel>
          <FormControl>
            <Input value={field.value as string} onChange={field.onChange} className="tracking-[0.5rem]" />
          </FormControl>
        </FormItem>
      )}
    />
  );
}

function DaysField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();
  const dayOptions = [
    { value: '0', label: t('Sunday') },
    { value: '1', label: t('Monday') },
    { value: '2', label: t('Tuesday') },
    { value: '3', label: t('Wednesday') },
    { value: '4', label: t('Thursday') },
    { value: '5', label: t('Friday') },
    { value: '6', label: t('Saturday') },
  ];

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.days` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="grid grid-cols-4">
          <FormLabel>{t('Days')}</FormLabel>
          <FormControl>
            <MultiSelect
              options={dayOptions}
              onValueChange={field.onChange}
              defaultValue={(field.value as string[]) || []}
              placeholder={t('Select days')}
              maxCount={3}
              useSearch={false}
              className="col-span-3"
            />
          </FormControl>
        </FormItem>
      )}
    />
  );
}

function BetweenField<T extends Schedule>({ form }: { form: UseFormReturn<T> }) {
  const { t } = useTranslation();

  return (
    <FormField
      control={form.control}
      name={`schedule_parameters.between` as FieldPath<T>}
      render={({ field }) => (
        <FormItem className="grid grid-cols-4">
          <FormLabel>{t('Between')}</FormLabel>
          <FormControl>
            <TimePicker
              mode="range"
              value={(field.value as string[]) || ['', '']}
              onChange={field.onChange}
              className="col-span-3 w-full"
            />
          </FormControl>
        </FormItem>
      )}
    />
  );
}
