import React, { useState } from 'react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Select, SelectContent, SelectGroup, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { useCreateSource, useEditSource } from '@/components/source/queries';
import type { Source } from './types';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { ScrollArea } from '@/components/ui/scroll-area';

// Helper function to extract lifetime value and unit for form display
const parseLifetime = (lifetime?: string) => {
  if (!lifetime) return { value: '24', unit: 'H' };
  const match = lifetime.match(/^(\d+)([A-Z])$/);
  if (match) {
    return { value: match[1], unit: match[2] };
  }
  return { value: '24', unit: 'H' };
};

const getSourceSchema = (existingSources: Source[], currentSource?: Source) => {
  const existingNames = existingSources.filter((s) => s.id !== currentSource?.id).map((s) => s.name.toLowerCase());

  const existingPorts = existingSources.filter((s) => s.id !== currentSource?.id).map((s) => s.port);

  return z.object({
    id: z.number().optional(),
    name: z
      .string()
      .min(1, 'Name is required')
      .refine((name) => !existingNames.includes(name.toLowerCase()), {
        message: 'A source with this name already exists',
      }),
    description: z.string().max(255, 'Description must be 255 characters or less').optional(),
    port: z
      .number()
      .min(1, 'Port is required')
      .max(65535, 'Port must be less than 65536')
      .refine((port) => !existingPorts.includes(port), { message: 'A source with this port already exists' }),
    flowtype: z.string().optional(),
    lifetimeValue: z.string().min(1, 'Lifetime is required'),
    lifetimeUnit: z.string().min(1, 'Choose a unit'),
    directory: z.string().optional(),
  });
};

type SourceValues = z.infer<ReturnType<typeof getSourceSchema>>;

export default function AddEditSourceDialog({
  source,
  sources = [],
  children,
  open,
  onOpenChange,
}: {
  source?: Source;
  sources?: Source[];
  children?: React.ReactNode;
  open?: boolean;
  onOpenChange?: (open: boolean) => void;
}) {
  const [internalOpen, setInternalOpen] = useState(false);
  const { t } = useTranslation();
  const createSource = useCreateSource();
  const updateSource = useEditSource();

  // Use external state if provided, otherwise use internal state
  const isOpen = open !== undefined ? open : internalOpen;
  const handleOpenChange = onOpenChange || setInternalOpen;

  const flowTypes = ['NetFlow', 'sFlow', 'jFlow', 'IPFIX'];
  const lifetimeUnits = [
    { key: 'H', label: t('Hours') },
    { key: 'D', label: t('Days') },
    { key: 'W', label: t('Weeks') },
  ];

  // Parse lifetime for form display
  const { value: lifetimeValue, unit: lifetimeUnit } = parseLifetime(source?.lifetime);

  const sourceSchema = getSourceSchema(sources, source);

  const form = useForm<SourceValues>({
    resolver: zodResolver(sourceSchema),
    defaultValues: {
      id: source?.id ?? undefined,
      name: source?.name ?? '',
      description: source?.description ?? '',
      port: source?.port ?? 0,
      flowtype: source?.flowtype ?? 'netflow',
      lifetimeValue: lifetimeValue,
      lifetimeUnit: lifetimeUnit,
      directory: source?.directory ?? '',
    },
  });

  const onSubmit: SubmitHandler<SourceValues> = (data) => {
    // Combine lifetime value and unit back into single field
    const submitData: Source = {
      ...data,
      id: data.id ?? -1,
      lifetime: `${data.lifetimeValue}${data.lifetimeUnit}`,
      port: Number(data.port),
    };

    if (source) {
      updateSource.mutate(submitData, {
        onSuccess: () => {
          handleOpenChange(false);
          form.reset();
        },
      });
    } else {
      createSource.mutate(submitData, {
        onSuccess: () => {
          handleOpenChange(false);
          form.reset();
        },
      });
    }
  };

  const handleDialogChange = (isOpen: boolean) => {
    handleOpenChange(isOpen);
    if (!isOpen) {
      form.reset();
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleDialogChange}>
      {children && <DialogTrigger asChild>{children}</DialogTrigger>}
      <DialogContent className="p-0 sm:max-w-[800px]">
        <DialogHeader className="bg-card p-6">
          <DialogTitle>{source ? t('Edit Flow Source') : t('Create Flow Source')}</DialogTitle>
          <DialogDescription>
            {t(
              'When adding a new source, make sure you set up the source to send flow data to your NNA installation IP address at the port you specify below to receive data.'
            )}
          </DialogDescription>
        </DialogHeader>
        <ScrollArea className="h-[550px]">
          <Form {...form}>
            <form id="source-form" onSubmit={form.handleSubmit(onSubmit)}>
              <div className="flex flex-col gap-4 p-6">
                <div className="mb-4">
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem className="flex w-full flex-wrap items-center gap-4">
                        <div className="space-y-1 md:w-2/3">
                          <FormLabel>{t('Flow Source Name')}</FormLabel>
                          <FormDescription>
                            {t('Name of the flow data collector, must be a unique name.')}
                          </FormDescription>
                        </div>
                        <div className="relative w-full flex-1">
                          <FormControl className="flex-1">
                            <div className="col-span-2 w-full">
                              <Input {...field} />
                            </div>
                          </FormControl>
                          <FormMessage className="absolute -bottom-6 left-0" />
                        </div>
                      </FormItem>
                    )}
                  />
                </div>
                <div className="mt-1 mb-4">
                  <FormField
                    control={form.control}
                    name="description"
                    render={({ field }) => (
                      <FormItem className="flex w-full flex-wrap items-center gap-4">
                        <div className="space-y-1 md:w-2/3">
                          <FormLabel>{t('Flow Source Description')}</FormLabel>
                          <FormDescription>
                            {t('Description of devices sending flow data to this source.')}
                          </FormDescription>
                        </div>
                        <div className="relative w-full flex-1">
                          <FormControl className="flex-1">
                            <div className="col-span-2 max-h-[200px] w-full">
                              <Textarea {...field} className="max-h-[200px]" maxLength={255} />
                            </div>
                          </FormControl>
                          <FormMessage />
                        </div>
                      </FormItem>
                    )}
                  />
                </div>
                <div className="mt-1 mb-4">
                  <FormField
                    control={form.control}
                    name="port"
                    render={({ field }) => (
                      <FormItem className="flex w-full flex-wrap items-center gap-4">
                        <div className="space-y-1 md:w-2/3">
                          <FormLabel>{t('Port')}</FormLabel>
                          <FormDescription>{t('Unique port that flow data is being recieved on.')}</FormDescription>
                        </div>
                        <div className="relative w-full flex-1">
                          <FormControl className="flex-1">
                            <Input
                              {...field}
                              type="number"
                              value={field.value || ''}
                              onChange={(e) => {
                                const val = e.target.value;
                                field.onChange(val === '' ? 0 : Number(val));
                              }}
                            />
                          </FormControl>
                          <FormMessage className="absolute -bottom-6 left-0" />
                        </div>
                      </FormItem>
                    )}
                  />
                </div>
                <div className="mt-1 mb-4">
                  <FormField
                    control={form.control}
                    name="flowtype"
                    render={({ field }) => (
                      <FormItem className="flex w-full flex-wrap items-center gap-4">
                        <FormLabel className="text-sm font-medium md:w-2/3">{t('Flow Type')}</FormLabel>
                        <FormControl className="flex-1">
                          <div className="col-span-2 w-full">
                            <Select
                              disabled={!!source}
                              value={field.value}
                              onValueChange={(selectedVal) => {
                                field.onChange(selectedVal);
                              }}
                            >
                              <SelectTrigger className="w-full">
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectGroup className="max-h-80 w-full">
                                  {flowTypes.map((flow, index) => (
                                    <SelectItem key={index} value={flow.toLowerCase()}>
                                      {flow}
                                    </SelectItem>
                                  ))}
                                </SelectGroup>
                              </SelectContent>
                            </Select>
                          </div>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
                <div className="mt-1 mb-4">
                  <FormItem className="flex w-full flex-wrap items-center gap-4">
                    <div className="space-y-1 md:w-2/3">
                      <FormLabel>{t('Raw Data Lifetime')}</FormLabel>
                      <FormDescription>{t('Length of time you want granular flow data to be stored.')}</FormDescription>
                    </div>
                    <FormControl className="flex-1">
                      <div className="flex items-center justify-between gap-4">
                        <FormField
                          control={form.control}
                          name="lifetimeValue"
                          render={({ field }) => (
                            <div className="relative w-full flex-1">
                              <FormItem className="flex-1">
                                <FormControl>
                                  <Input type="number" min={1} {...field} />
                                </FormControl>
                                <FormMessage className="absolute -bottom-6 left-0" />
                              </FormItem>
                            </div>
                          )}
                        />
                        <FormField
                          control={form.control}
                          name="lifetimeUnit"
                          render={({ field }) => (
                            <Select value={field.value} onValueChange={field.onChange}>
                              <SelectTrigger>
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                {lifetimeUnits.map(({ key, label }) => (
                                  <SelectItem key={key} value={key}>
                                    {label}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          )}
                        />
                      </div>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                </div>
                <div className="mt-1">
                  <FormField
                    control={form.control}
                    name="directory"
                    render={({ field }) => (
                      <FormItem className="flex w-full flex-wrap items-center gap-4">
                        <div className="space-y-1 md:w-2/3">
                          <FormLabel className="text-sm font-medium md:w-2/3">{t('Data Directory')}</FormLabel>
                          <FormDescription>
                            {t(
                              "The directory where the source's flow data is stored. The directory location requires nna:users ownership and 775 permissions to be written to. A directory will be created with the source id inside the directory you specify."
                            )}
                          </FormDescription>
                        </div>
                        <div className="relative w-full flex-1">
                          <FormControl className="flex-1">
                            <div className="col-span-2 w-full">
                              <Input {...field} placeholder="/usr/local/nagiosna/var" disabled={!!source} />
                            </div>
                          </FormControl>
                          <FormMessage className="absolute -bottom-6 left-0" />
                        </div>
                      </FormItem>
                    )}
                  />
                </div>
              </div>
            </form>
          </Form>
        </ScrollArea>
        <DialogFooter className="bg-card p-6">
          <Button type="button" variant="secondary" onClick={() => handleDialogChange(false)}>
            {t('Cancel')}
          </Button>
          <Button type="submit" form="source-form">
            {source ? t('Update Flow Source') : t('Create Flow Source')}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
