import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '@/components/ui/dropdown-menu';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import { Button } from '@/components/ui/button';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import { ColumnDef } from '@tanstack/react-table';
import { Ellipsis, Pencil, Trash2, CirclePause, CirclePower, RotateCcw } from 'lucide-react';
import { Source } from '@/components/source/types';
import { useTranslation } from 'react-i18next';
import { useState } from 'react';
import { SourceTraffic } from '@/components/source/SourceTraffic';
import { Link } from 'react-router-dom';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import AddEditSourceDialog from '@/components/source/AddEditSourceDialog';
import { useAuth } from '@/contexts/AuthContext';
import { useStartSource, useStopSource, useRestartSource } from '@/components/source/queries';
import { hasPermission } from '@/components/role/utils';
import { Checkbox } from '@/components/ui/checkbox';
import DeleteSourcesDialog from '@/components/source/DeleteSourcesDialog';

type SourcesDataTableProps = {
  sources: Source[];
  isLoading?: boolean;
};

export default function SourcesDataTable({ sources, isLoading }: SourcesDataTableProps) {
  const { t } = useTranslation();
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingSource, setEditingSource] = useState<Source | undefined>(undefined);

  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [idsToDelete, setIdsToDelete] = useState<number[]>([]);

  const handleDelete = (id: number) => {
    setIdsToDelete([id]);
    setDeleteDialogOpen(true);
  };

  const { user } = useAuth();

  const startSourceMutation = useStartSource();
  const stopSourceMutation = useStopSource();
  const restartSourceMutation = useRestartSource();

  const actions = [];
  if (hasPermission(user?.role, 'flow_source_permissions', 'sources', 'delete')) {
    actions.push({
      label: t('Delete'),
      value: 'delete',
      onClick: async (selectedRows: Source[]) => {
        setIdsToDelete(selectedRows.map((row) => row.id!));
        setDeleteDialogOpen(true);
      },
    });
  }

  const columns: ColumnDef<Source>[] = [
    {
      id: 'select',
      header: ({ table }) => (
        <Checkbox
          checked={table.getIsAllPageRowsSelected() || (table.getIsSomePageRowsSelected() && 'indeterminate')}
          onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
          aria-label="Select all"
        />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
    },
    {
      accessorKey: 'status',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t('Status')} />,
      cell: ({ row }) => (
        <div className="justify-left ml-5 flex">
          <Tooltip>
            <TooltipTrigger asChild>
              <span
                className={`inline-block h-3 w-3 rounded-full ${row.original.status ? 'bg-success' : 'bg-error'}`}
              />
            </TooltipTrigger>
            <TooltipContent>
              <p>{row.original.status ? t('Active') : t('Disabled')}</p>
            </TooltipContent>
          </Tooltip>
        </div>
      ),
    },
    {
      accessorKey: 'name',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Flow Source Name" />,
      cell: ({ row }) => (
        <Link className="text-primary" to={`/sources/${row.original.id}`}>
          {row.original.name}
        </Link>
      ),
    },
    {
      accessorKey: 'traffic',
      id: 'traffic',
      header: t('Traffic Last 30 Minutes'),
      cell: ({ row }) => {
        const arr = row.original.traffic;
        let maxValue: number | null = null;

        if (Array.isArray(arr)) {
          const values = arr.map((a) => a[0]).filter((v) => v != null);
          if (values.length > 0) {
            maxValue = Math.max(...values);
          }
        }

        return Array.isArray(arr) && maxValue !== 0 && maxValue !== null ? (
          <SourceTraffic data={arr} maxValue={maxValue} />
        ) : (
          <span>{t('No Data')}</span>
        );
      },
    },
    {
      accessorKey: 'diskusage',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t('Disk')} />,
      cell: ({ row }) => <span>{row.original.diskusage || t('No Data')}</span>,
    },
    {
      accessorKey: 'lifetime',
      header: t('Data Lifetime'),
    },
    {
      accessorKey: 'flowtype',
      header: t('Flow Type'),
    },
    {
      accessorKey: 'port',
      header: ({ column }) => <DataTableColumnHeader column={column} title="Port" />,
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => {
        const canStartStop = hasPermission(user?.role, 'flow_source_permissions', 'start_stop_sources');
        const canEdit = hasPermission(user?.role, 'flow_source_permissions', 'sources', 'put');
        const canDelete = hasPermission(user?.role, 'flow_source_permissions', 'sources', 'delete');
        const canShowMenu = canStartStop || canEdit || canDelete;

        return (
          <>
            {canShowMenu && (
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost">
                    <Ellipsis />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  {canStartStop && (
                    <>
                      {!row.original.status && (
                        <DropdownMenuItem onClick={() => startSourceMutation.mutate(row.original.id!)}>
                          <CirclePower />
                          {t('Start')}
                        </DropdownMenuItem>
                      )}
                      {row.original.status && (
                        <DropdownMenuItem onClick={() => stopSourceMutation.mutate(row.original.id!)}>
                          <CirclePause />
                          {t('Stop')}
                        </DropdownMenuItem>
                      )}
                      {row.original.status && (
                        <DropdownMenuItem onClick={() => restartSourceMutation.mutate(row.original.id!)}>
                          <RotateCcw />
                          {t('Restart')}
                        </DropdownMenuItem>
                      )}
                    </>
                  )}
                  {canEdit && (
                    <DropdownMenuItem
                      onClick={() => {
                        setEditingSource(row.original);
                        setEditDialogOpen(true);
                      }}
                    >
                      <Pencil />
                      {t('Edit')}
                    </DropdownMenuItem>
                  )}
                  {canDelete && (
                    <DropdownMenuItem onClick={() => handleDelete(row.original.id!)}>
                      <Trash2 />
                      {t('Delete')}
                    </DropdownMenuItem>
                  )}
                </DropdownMenuContent>
              </DropdownMenu>
            )}
          </>
        );
      },
    },
  ];

  return (
    <>
      {isLoading ? (
        <DataTableSkeleton></DataTableSkeleton>
      ) : (
        <DataTable columns={columns} data={sources} usesRowSelection={true} selectedRowActions={actions} />
      )}
      <DeleteSourcesDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      />
      {editingSource && (
        <AddEditSourceDialog
          source={editingSource}
          sources={sources}
          open={editDialogOpen}
          onOpenChange={(open) => {
            setEditDialogOpen(open);
            if (!open) setEditingSource(undefined);
          }}
        />
      )}
    </>
  );
}
