import { useTranslation } from 'react-i18next';
import type { Source } from '@/components/source/types';
import { useState } from 'react';
import { useGetSourceSummaryChart, useGetSourceSummaryTalkers } from '@/components/source/queries';
import { useGetSourceGroupSummaryChart, useGetSourceGroupSummaryTalkers } from '@/components/sourcegroups/queries';
import { TopTalkersSummary } from '@/components/source/TopTalkersSummary';
import { SummaryChart } from '@/components/source/SummaryChart';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectGroup, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { PageSectionTitle } from '@/components/Page';

export function Summary({ data, isSourceGroup = false }: { data: Source; view?: string; isSourceGroup?: boolean }) {
  const { t } = useTranslation();
  const [timeframe, setTimeframe] = useState('-2 hours');

  const { data: groupChart, isLoading: isGroupChartLoading } = useGetSourceGroupSummaryChart(data?.id ?? 0, timeframe, {
    enabled: isSourceGroup,
  });

  const { data: groupTalkers, isLoading: isGroupTalkersLoading } = useGetSourceGroupSummaryTalkers(
    data?.id ?? 0,
    timeframe,
    { enabled: isSourceGroup }
  );

  const { data: sourceChart, isLoading: isSourceChartLoading } = useGetSourceSummaryChart(data?.id ?? 0, timeframe, {
    enabled: !isSourceGroup,
  });

  const { data: sourceTalkers, isLoading: isSourceTalkersLoading } = useGetSourceSummaryTalkers(
    data?.id ?? 0,
    timeframe,
    { enabled: !isSourceGroup }
  );

  const summaryChart = isSourceGroup ? groupChart : sourceChart;
  const summaryTalkers = isSourceGroup ? groupTalkers : sourceTalkers;
  const isChartLoading = isSourceGroup ? isGroupChartLoading : isSourceChartLoading;
  const isLoading = isSourceGroup ? isGroupTalkersLoading : isSourceTalkersLoading;

  const [isLogarithmic, setIsLogarithmic] = useState(true);

  const rawChartData = summaryChart?.chart_data || [];

  return (
    <div className="flex flex-col gap-6">
      <Card className="pt-0">
        <CardHeader className="flex items-center gap-2 border-b py-4 sm:flex-row">
          <div className="grid flex-1 gap-1">
            <CardTitle>{t('Bandwidth Data')}</CardTitle>
            <CardDescription className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                {t('Showing data for the')}
                <Select
                  value={timeframe}
                  onValueChange={(value) => {
                    setTimeframe(value);
                  }}
                >
                  <SelectTrigger className="w-[180px]">
                    <SelectValue placeholder="Timeframe" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectGroup>
                      <SelectItem value="-2 hours">{t('Last 2 Hours')}</SelectItem>
                      <SelectItem value="-4 hours">{t('Last 4 Hours')}</SelectItem>
                      <SelectItem value="-6 hours">{t('Last 6 Hours')}</SelectItem>
                      <SelectItem value="-12 hours">{t('Last 12 Hours')}</SelectItem>
                      <SelectItem value="-24 hours">{t('Last 24 Hours')}</SelectItem>
                      <SelectItem value="-2 days">{t('Last 2 Days')}</SelectItem>
                      <SelectItem value="-1 week">{t('Last Week')}</SelectItem>
                      <SelectItem value="-1 month">{t('Last Month')}</SelectItem>
                    </SelectGroup>
                  </SelectContent>
                </Select>
              </div>
              <div className="border-muted flex gap-1 overflow-hidden rounded-md border p-1">
                <button
                  onClick={() => setIsLogarithmic(false)}
                  className={`w-[120px] rounded-md px-3 py-1 text-sm font-medium transition-all transition-colors duration-300 ease-out ${
                    !isLogarithmic
                      ? 'bg-primary text-primary-foreground'
                      : 'bg-card text-muted-foreground hover:bg-muted'
                  }`}
                >
                  {t('Linear')}
                </button>
                <button
                  onClick={() => setIsLogarithmic(true)}
                  className={`w-[120px] rounded-md px-3 py-1 text-sm font-medium transition-all transition-colors duration-300 ease-out ${
                    isLogarithmic
                      ? 'bg-primary text-primary-foreground'
                      : 'bg-card text-muted-foreground hover:bg-muted'
                  }`}
                >
                  {t('Logarithmic')}
                </button>
              </div>
            </CardDescription>
          </div>
        </CardHeader>
        <CardContent className="px-2 sm:px-6">
          <SummaryChart data={rawChartData} isLoading={isChartLoading} isLogarithmic={isLogarithmic} />
        </CardContent>
      </Card>
      <PageSectionTitle>{t('Top 5 Talkers')}</PageSectionTitle>
      <TopTalkersSummary summaryTalkers={summaryTalkers ?? {}} isLoading={isLoading} />
    </div>
  );
}
