import { useState, useMemo, useEffect, useRef } from 'react';
import { useTranslation } from 'react-i18next';
import { Area, CartesianGrid, ComposedChart, ReferenceArea, ResponsiveContainer, XAxis, YAxis } from 'recharts';
import type { SummaryDataPoint, RechartsMouseEvent } from './types';
import { ChartContainer, ChartTooltip, ChartTooltipContent } from '@/components/ui/chart';
import { Button } from '@/components/ui/button';

type SummaryChartProps = {
  data: SummaryDataPoint[];
  isLoading: boolean;
  isLogarithmic: boolean;
  bytesUnit?: string;
  onResetZoom?: () => void;
};

export function SummaryChart({ data, isLoading, isLogarithmic, onResetZoom }: SummaryChartProps) {
  const { t } = useTranslation();

  const [visibleLines, setVisibleLines] = useState({
    bytes: true,
    flows: true,
    packets: true,
    bps: true,
  });

  const [refAreaLeft, setRefAreaLeft] = useState<string | null>(null);
  const [refAreaRight, setRefAreaRight] = useState<string | null>(null);
  const [startTime, setStartTime] = useState<string | null>(null);
  const [endTime, setEndTime] = useState<string | null>(null);
  const [isSelecting, setIsSelecting] = useState(false);

  const formatBytes = (bytes: number) => {
    if (bytes === 0) return 0;
    const k = 1000;
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2));
  };

  const getBytesUnit = (bytes: number) => {
    if (bytes === 0) return 'B';
    const k = 1000;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return sizes[i];
  };

  const originalChartData = data.map((item: SummaryDataPoint) => ({
    ...item,
    formattedBytes: formatBytes(item.bytes),
  }));

  // Zoom functionality - filter data based on selected time range
  const zoomedData = useMemo(() => {
    if (!startTime || !endTime || originalChartData.length === 0) {
      return originalChartData;
    }

    const dataPointsInRange = originalChartData.filter(
      (dataPoint: SummaryDataPoint) => dataPoint.date >= startTime && dataPoint.date <= endTime
    );

    // Ensure we have at least two data points for the chart
    return dataPointsInRange.length > 1 ? dataPointsInRange : originalChartData.slice(0, 2);
  }, [startTime, endTime, originalChartData]);

  const previousDataRef = useRef<SummaryDataPoint[]>([]);

  // reset zoom if data timeframe is changed
  useEffect(() => {
    const prev = previousDataRef.current;
    const isDifferent = prev.length !== data.length || prev[0]?.date !== data[0]?.date;

    if (isDifferent) {
      setStartTime(null);
      setEndTime(null);
      previousDataRef.current = data;
    }
  }, [data]);

  const chartConfig = {
    bytes: {
      label: t(`Bytes`),
      color: 'var(--chart-1)',
    },
    flows: {
      label: t('Flows'),
      color: 'var(--chart-2)',
    },
    packets: {
      label: t('Packets'),
      color: 'var(--chart-3)',
    },
    bps: {
      label: t('Bits/Second'),
      color: 'var(--chart-4)',
    },
  };

  const handleMouseDown = (e: RechartsMouseEvent) => {
    if (e.activeLabel) {
      setRefAreaLeft(e.activeLabel);
      setIsSelecting(true);
    }
  };

  const handleMouseMove = (e: RechartsMouseEvent) => {
    if (isSelecting && e.activeLabel) {
      setRefAreaRight(e.activeLabel);
    }
  };

  const handleMouseUp = () => {
    if (refAreaLeft && refAreaRight) {
      const [left, right] = [refAreaLeft, refAreaRight].sort();
      const minTimeDiff = 5 * 60 * 1000;

      if (new Date(right).getTime() - new Date(left).getTime() >= minTimeDiff) {
        setStartTime(left);
        setEndTime(right);
      }
    }
    setRefAreaLeft(null);
    setRefAreaRight(null);
    setIsSelecting(false);
  };

  const handleReset = () => {
    setStartTime(null);
    setEndTime(null);
    if (onResetZoom) onResetZoom();
  };

  return (
    <div className="flex flex-col gap-4">
      <div className="flex flex-wrap items-center justify-between gap-2">
        <div className="flex flex-wrap gap-2">
          {Object.entries(chartConfig).map(([key, config]) => (
            <button
              key={key}
              onClick={() =>
                setVisibleLines((prev) => ({
                  ...prev,
                  [key]: !prev[key as keyof typeof visibleLines],
                }))
              }
              className={`flex items-center gap-2 rounded-md px-3 py-1 text-sm font-medium transition-colors ${
                visibleLines[key as keyof typeof visibleLines]
                  ? 'bg-primary/10 text-primary border-primary/20 border'
                  : 'bg-muted text-muted-foreground border-muted border'
              }`}
            >
              <div
                className="h-3 w-3 rounded-sm"
                style={{
                  backgroundColor: visibleLines[key as keyof typeof visibleLines] ? config.color : 'transparent',
                }}
              />
              {config.label}
            </button>
          ))}
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={handleReset}
            disabled={!startTime && !endTime}
            className="text-xs sm:text-sm"
          >
            {t('Reset Zoom')}
          </Button>
        </div>
      </div>
      <ChartContainer config={chartConfig} className="h-[250px] w-full">
        <ResponsiveContainer width="100%" height="100%">
          {isLoading ? (
            <div className="text-muted-foreground flex h-[200px] items-center justify-center">
              {t('Loading chart data...')}
            </div>
          ) : zoomedData.length > 0 ? (
            <ComposedChart
              data={zoomedData}
              margin={{ top: 10, right: 10, left: 0, bottom: 0 }}
              onMouseDown={handleMouseDown}
              onMouseMove={handleMouseMove}
              onMouseUp={handleMouseUp}
              onMouseLeave={handleMouseUp}
            >
              <defs>
                {Object.keys(chartConfig).map((key) => (
                  <linearGradient key={key} id={`fill${key}`} x1="0" y1="0" x2="0" y2="1">
                    <stop offset="5%" stopColor={`var(--color-${key})`} stopOpacity={0.8} />
                    <stop offset="95%" stopColor={`var(--color-${key})`} stopOpacity={0.1} />
                  </linearGradient>
                ))}
              </defs>
              <CartesianGrid vertical={false} />
              <XAxis
                dataKey="date"
                tickLine={false}
                axisLine={false}
                tickMargin={8}
                minTickGap={32}
                tickFormatter={(value) =>
                  new Date(value).toLocaleTimeString('en-US', {
                    hour: 'numeric',
                    minute: '2-digit',
                    hour12: false,
                  })
                }
              />
              <YAxis
                scale={isLogarithmic ? 'log' : 'linear'}
                domain={isLogarithmic ? ['dataMin', 'dataMax'] : ['auto', 'auto']}
                tickLine={false}
                axisLine={false}
                tickMargin={8}
                tickFormatter={(value) =>
                  value >= 1_000_000
                    ? `${(value / 1_000_000).toFixed(1)}M`
                    : value >= 1_000
                      ? `${(value / 1_000).toFixed(1)}K`
                      : value.toString()
                }
              />
              <ChartTooltip
                cursor={false}
                content={
                  <ChartTooltipContent
                    labelFormatter={(value) =>
                      new Date(value).toLocaleString('en-US', {
                        month: 'numeric',
                        day: 'numeric',
                        year: 'numeric',
                        hour: 'numeric',
                        minute: '2-digit',
                        hour12: false,
                      })
                    }
                    formatter={(value, name) => {
                      const config = chartConfig[name as keyof typeof chartConfig];
                      if (name === 'bytes') {
                        const numValue = Number(value);
                        const formatted = formatBytes(numValue);
                        const unit = getBytesUnit(numValue);
                        return [
                          <span key={name} className="flex min-w-[120px] justify-between">
                            <span>
                              {t('Bytes')} ({unit})
                            </span>
                            <span>{formatted}</span>
                          </span>,
                        ];
                      }
                      return [
                        <span key={name} className="flex min-w-[120px] justify-between">
                          <span>{config?.label || name}</span>
                          <span>{value}</span>
                        </span>,
                      ];
                    }}
                    indicator="dot"
                  />
                }
              />
              {Object.keys(chartConfig).map((key) =>
                visibleLines[key as keyof typeof visibleLines] ? (
                  <Area
                    key={key}
                    dataKey={key}
                    type="monotone"
                    fill={`url(#fill${key})`}
                    stroke={`var(--color-${key})`}
                    fillOpacity={0.6}
                    isAnimationActive={false}
                  />
                ) : null
              )}
              {refAreaLeft && refAreaRight && (
                <ReferenceArea
                  x1={refAreaLeft}
                  x2={refAreaRight}
                  strokeOpacity={0.8}
                  fill="hsl(var(--primary))"
                  fillOpacity={0.35}
                />
              )}
            </ComposedChart>
          ) : (
            <div className="text-muted-foreground flex h-[200px] items-center justify-center">
              {t('No data available')}
            </div>
          )}
        </ResponsiveContainer>
      </ChartContainer>
    </div>
  );
}
