import { useTranslation } from 'react-i18next';
import type { TopTalkersSummaryProps } from './types';
import { Skeleton } from '../ui/skeleton';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Card, CardContent } from '@/components/ui/card';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { GlobeLockIcon, MoreHorizontal } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useState } from 'react';
import SearchSuricata from '@/components/suricata/SearchSuricata';
import SearchRingBufferDialog from '@/components/wireshark/ringbuffer/SearchRingBufferDialog';
import SharkFin from '@/components/ui/icons/SharkFin';
import { useGetIntegrations } from '../queries';

export function TopTalkersSummary({ summaryTalkers, isLoading }: TopTalkersSummaryProps) {
  const { t } = useTranslation();
  const [searchSuricataDialogOpen, setSearchSuricataDialogOpen] = useState(false);
  const [searchWiresharkDialogOpen, setSearchWiresharkDialogOpen] = useState(false);
  const [activeTalker, setActiveTalker] = useState<{ value: string; type: string } | null>(null);
  const { data: integrationStatus } = useGetIntegrations();

  const talkerSections = [
    { key: 'dstip', label: t('Destination IP') },
    { key: 'srcip', label: t('Source IP') },
    { key: 'dstport', label: t('Dest. Port') },
    { key: 'srcport', label: t('Src. Port') },
  ];

  return (
    <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:grid-cols-4">
      {/* TODO: top talkers data can over flow cards on screen sizes, Fix responsiveness */}
      {talkerSections.map((section) => {
        const talkerList = summaryTalkers?.[section.key];
        return (
          <Card key={section.key}>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>{section.label}</TableHead>
                    <TableHead className="text-right">% {t('Bytes')}</TableHead>
                    <TableHead></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {isLoading ? (
                    Array.from({ length: 5 }).map((_, idx) => (
                      <TableRow key={idx}>
                        <TableCell colSpan={2}>
                          <Skeleton className="h-5 w-full" />
                        </TableCell>
                      </TableRow>
                    ))
                  ) : talkerList && talkerList.length > 1 ? (
                    talkerList.slice(1).map((talker, idx) => (
                      <TableRow key={idx}>
                        <TableCell>{talker[section.key]}</TableCell>
                        <TableCell className="text-right">{talker.bytespercent}</TableCell>
                        {(!!integrationStatus?.suricata ||
                          (!!integrationStatus?.wireshark && !!integrationStatus?.ringbuffer)) && (
                          <TableCell>
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button variant="ghost" className="h-8 w-8 p-0">
                                  <span className="sr-only">{t('Open menu')}</span>
                                  <MoreHorizontal className="h-4 w-4" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="center">
                                {!!integrationStatus?.suricata && (
                                  <DropdownMenuItem
                                    onSelect={() => {
                                      const type = section.key.includes('ip') ? 'ip' : 'port';
                                      setActiveTalker({ value: talker[section.key], type });
                                      setSearchSuricataDialogOpen(true);
                                    }}
                                  >
                                    <GlobeLockIcon />
                                    {t('Search Suricata')}
                                  </DropdownMenuItem>
                                )}
                                {!!integrationStatus?.wireshark && integrationStatus?.ringbuffer && (
                                  <DropdownMenuItem
                                    onSelect={() => {
                                      const type = section.key.includes('ip') ? 'ip' : 'port';
                                      setActiveTalker({ value: talker[section.key], type });
                                      setSearchWiresharkDialogOpen(true);
                                    }}
                                  >
                                    <SharkFin />
                                    {t('Search')} Wireshark
                                  </DropdownMenuItem>
                                )}
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </TableCell>
                        )}
                      </TableRow>
                    ))
                  ) : (
                    <TableRow>
                      <TableCell colSpan={2} className="text-muted-foreground text-center">
                        {t('No data available')}
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        );
      })}

      <SearchSuricata
        open={searchSuricataDialogOpen}
        onOpenChange={setSearchSuricataDialogOpen}
        srcIp={activeTalker?.type === 'ip' ? activeTalker.value : undefined}
        destIp={activeTalker?.type === 'ip' ? activeTalker.value : undefined}
        srcPort={activeTalker?.type === 'port' ? Number(activeTalker.value) : undefined}
        destPort={activeTalker?.type === 'port' ? Number(activeTalker.value) : undefined}
      />
      <SearchRingBufferDialog
        open={searchWiresharkDialogOpen}
        onOpenChange={setSearchWiresharkDialogOpen}
        src={activeTalker?.type === 'ip' ? activeTalker.value : undefined}
        dst={activeTalker?.type === 'ip' ? activeTalker.value : undefined}
        srcPort={activeTalker?.type === 'port' ? Number(activeTalker.value) : undefined}
        dstPort={activeTalker?.type === 'port' ? Number(activeTalker.value) : undefined}
      />
    </div>
  );
}
