import { keepPreviousData, useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import { Source } from './types';
import { MultiSourceDataOptions } from '../dashboard/types';

export function useGetSources() {
  return useQuery<Source[], AxiosError>({
    queryKey: ['Sources'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/sources');
      return response.data;
    },
  });
}

export function useGetSource(id: number) {
  return useQuery<Source, AxiosError>({
    queryKey: ['Sources', id],
    queryFn: async () => {
      const response = await axios.get('/api/v1/sources/' + id);
      return response.data;
    },
  });
}

export function useGetSourceSummaryChart(id: number, timeframe: string, props = {}) {
  return useQuery({
    queryKey: ['dashlet', 'source_summary', id, timeframe],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/sources/${id}/summary/chart`, {
        params: { timeframe },
      });
      return response.data;
    },
    ...props,
    placeholderData: keepPreviousData,
  });
}

export function useGetSourcesSummaryChart(timeframe: string, options: MultiSourceDataOptions, props = {}) {
  return useQuery({
    queryKey: ['dashlet', 'sources_summary', timeframe, options],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/sources/summary/chart`, {
        params: {
          timeframe,
          ...options,
        },
      });
      return response.data;
    },
    ...props,
    placeholderData: keepPreviousData,
  });
}

export function useGetSourceSummaryTalkers(id: number, timeframe: string, props = {}) {
  return useQuery({
    queryKey: ['SummaryTalkers', id, timeframe],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/sources/${id}/summary/talkers`, {
        params: { timeframe },
      });
      return response.data;
    },
    ...props,
  });
}

export function useCreateSource() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Source) => {
      return axios.post('/api/v1/sources', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
      toast(t('Source successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create source.'), { description: t('Please try again.') });
    },
  });
}

export function useEditSource() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Source) => {
      return axios.put(`/api/v1/sources/${data.id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
      toast(t('Source has been edited.'));
    },
    onError: () => {
      toast(t('Failed to edit source.'), {
        description: t('Please try again.'),
      });
    },
  });
}

export function useDeleteSources() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (ids: number | number[]) => {
      const idArray = Array.isArray(ids) ? ids : [ids];
      return axios.delete('/api/v1/sources', { data: { ids: idArray } });
    },
    onMutate: async (ids) => {
      const idArray = Array.isArray(ids) ? ids : [ids];
      const key = ['Sources'];
      await queryClient.cancelQueries({ queryKey: key });
      const previousSources = queryClient.getQueryData<Source[]>(key);
      queryClient.setQueryData(key, (old: Source[] = []) => old.filter((source) => !idArray.includes(source.id)));
      return { previousSources };
    },
    onSuccess: (_data, ids) => {
      const idArray = Array.isArray(ids) ? ids : [ids];
      const message = idArray.length > 1 ? t('Sources deleted.') : t('Source deleted.');
      toast(message);
    },
    onError: (_error, ids, context) => {
      queryClient.setQueryData(['Sources'], context?.previousSources);
      const idArray = Array.isArray(ids) ? ids : [ids];
      const errorMessage = idArray.length > 1 ? t('Failed to delete sources.') : t('Failed to delete source.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
    },
  });
}

export function useStartSource() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => axios.post(`/api/v1/sources/${id}/start`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
      toast(t('Source has successfully started.'));
    },
    onError: () => {
      toast(t('Failed to start source.'), { description: t('Please try again') });
    },
  });
}

export function useStopSource() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => axios.post(`/api/v1/sources/${id}/stop`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
      toast(t('Source has successfully stopped.'));
    },
    onError: () => {
      toast(t('Failed to stop source.'), { description: t('Please try again') });
    },
  });
}

export function useRestartSource() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => axios.post(`/api/v1/sources/${id}/restart`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['Sources'] });
      toast(t('Source has successfully restarted.'));
    },
    onError: () => {
      toast(t('Failed to restart source.'), { description: t('Please try again') });
    },
  });
}
