import { useTranslation } from 'react-i18next';
import { useForm } from 'react-hook-form';
import { useState, useEffect } from 'react';
import type { Source, QueryParam, QueryResults, FormParam, TimeRangeKey } from '@/components/source/types';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import { columns } from '@/components/source/queries/columns';
import { Button } from '@/components/ui/button';
import { PageSeparator } from '@/components/Page';
import QueryDropdown from '@/components/source/queries/QueryDropdown';
import { useRunQuery } from './queries';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { AxiosError } from '@/lib/axios';
import { Card, CardContent, CardTitle, CardHeader, CardFooter } from '@/components/ui/card';
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { QuerySummaryTable, QuerySummarySkeleton } from '@/components/source/queries/QuerySummaryTable';
import { QueryChart, QueryChartSkeleton } from '@/components/source/queries/QueryChart';

export function Queries({ src }: { src: Source[] | FormParam }) {
  const { t } = useTranslation();

  const isFormParam = !Array.isArray(src);

  const [currentQuery, setCurrentQuery] = useState<QueryParam | null>(isFormParam ? src : null);
  const { data, isLoading, isRefetching, error } = useRunQuery(currentQuery);

  const formDefaults: FormParam = isFormParam
    ? { ...src }
    : {
        sources: src,
        raw_query: '',
        flow_start: '-2 hour',
        time_range: '2h',
      };

  const form = useForm<FormParam>({
    defaultValues: formDefaults,
  });

  useEffect(() => {
    if (error) {
      const axiosError = error as AxiosError<{ error: string }>;
      const message =
        axiosError.response?.data?.error ??
        t(`Something has gone wrong with this query. The memory limit has likely been exceeded.
                Try searching a smaller timeframe, or using specific IPs or protocols.
                Check logs at /var/www/html/nagiosna/storage/logs/laravel.log for more information.`);
      form.setError('raw_query', { message });
    } else {
      form.clearErrors('raw_query');
    }
  }, [error, form, t]);

  const timeRange = form.watch('time_range');

  const presetTimeRanges: Record<
    Exclude<TimeRangeKey, 'custom elapsed time'>,
    { flow_start: string; flow_end: string }
  > = {
    '2h': { flow_start: '-2 hour', flow_end: '-1 second' },
    '4h': { flow_start: '-4 hour', flow_end: '-1 second' },
    '6h': { flow_start: '-6 hour', flow_end: '-1 second' },
    '12h': { flow_start: '-12 hour', flow_end: '-1 second' },
    '24h': { flow_start: '-24 hour', flow_end: '-1 second' },
    '2d': { flow_start: '-2 day', flow_end: '-1 second' },
    '1w': { flow_start: '-1 week', flow_end: '-1 second' },
    '1m': { flow_start: '-1 month', flow_end: '-1 second' },
  };

  const getTimeValues = (start: string, end?: string) => {
    if (timeRange in presetTimeRanges) {
      return presetTimeRanges[timeRange as keyof typeof presetTimeRanges];
    } else if (timeRange === 'custom elapsed time') {
      return {
        flow_start: start,
        flow_end: '-1 second',
      };
    }
    return {
      flow_start: start,
      flow_end: end,
    };
  };

  const handleRowClick = (row: QueryResults) => {
    const formValues = form.getValues();
    const sources = formValues.sources;
    const timeValues = getTimeValues(formValues.flow_start, formValues.flow_end);

    const newQuery = {
      sources,
      ...timeValues,
      raw_query: `${formValues.raw_query.trim() !== '' ? `(${formValues.raw_query.trim()}) and ` : ''}(src ip ${row.srcip} and dst ip ${row.dstip} and src port ${row.srcport} and dst port ${row.dstport})`,
    };

    form.setValue('raw_query', newQuery.raw_query);
    setCurrentQuery(newQuery);
  };

  const onSubmit = (formParam: FormParam) => {
    const { time_range, ...rest } = formParam;
    const timeValues = getTimeValues(formParam.flow_start, formParam.flow_end);

    setCurrentQuery({ ...rest, ...timeValues });
  };

  return (
    <>
      <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:grid-cols-3">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <Card className="flex h-full flex-col">
              <CardContent>
                <FormField
                  control={form.control}
                  name="time_range"
                  render={({ field }) => (
                    <FormItem className="flex flex-wrap items-center justify-between gap-4">
                      <div className="space-y-1">
                        <FormLabel>{t('Time Frame')}</FormLabel>
                        <FormDescription>{t('Set the time frame of the query')}</FormDescription>
                      </div>
                      <FormControl>
                        <div className="w-[200px]">
                          <QueryDropdown {...field} />
                        </div>
                      </FormControl>
                    </FormItem>
                  )}
                />
                {timeRange == 'custom elapsed time' && (
                  <FormField
                    control={form.control}
                    name="flow_start"
                    render={({ field }) => (
                      <FormItem className="flex flex-wrap items-center justify-between gap-4 pt-2">
                        <div className="space-y-1">
                          <FormDescription>{t('From: ')}</FormDescription>
                        </div>
                        <FormControl>
                          <div className="w-[200px]">
                            <Input placeholder={t('Flow Start')} {...field} value={field.value} />
                          </div>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                )}
                <PageSeparator />
                <FormItem className="flex flex-wrap items-center justify-between gap-4">
                  <div className="space-y-1">
                    <FormLabel>{t('Raw Query')}</FormLabel>
                    <FormDescription>{t('Uses nfdump raw query formating')}</FormDescription>
                  </div>
                  <div className="flex w-md gap-2">
                    <FormField
                      control={form.control}
                      name="raw_query"
                      render={({ field }) => (
                        <FormItem className="w-full">
                          <FormControl>
                            <Textarea {...field} className="col-span-3" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                </FormItem>
              </CardContent>
              <CardFooter>
                <Button type="submit">Run Query</Button>
              </CardFooter>
            </Card>
          </form>
        </Form>
        {((data && !isRefetching) || isLoading || isRefetching) && (
          <>
            {data && !isRefetching && <QueryChart summary={data.summary} results={data.results} />}
            {(isRefetching || isLoading) && <QueryChartSkeleton />}
            <Card>
              <CardHeader>
                <CardTitle>{t('Summary Data')}</CardTitle>
              </CardHeader>
              <CardContent>
                {data && !isRefetching && <QuerySummaryTable {...data.summary} />}
                {(isRefetching || isLoading) && <QuerySummarySkeleton />}
              </CardContent>
            </Card>
          </>
        )}
      </div>
      <div className="pt-8">
        {data && !isRefetching && (
          <DataTable
            columns={columns()}
            data={data.results}
            onRowClick={handleRowClick}
            sorting={[{ id: 'Flow Start', desc: false }]}
            usesRowSelection={false}
            selectedRowActions={[]}
            defaultColumnVisibility={{
              'Flow End': false,
              'Bits/Second': false,
              'Packets/Second': false,
              'Bytes/Packet': false,
            }}
          />
        )}
        {(isLoading || isRefetching) && <DataTableSkeleton />}
      </div>
    </>
  );
}
