import React, { useState, useMemo } from 'react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { useCreateSourceGroup, useEditSourceGroup } from '@/components/sourcegroups/queries';
import { SourceGroup } from '@/components/sourcegroups/types';
import { Source } from '@/components/source/types';
import { Textarea } from '@/components/ui/textarea';
import { useGetSources } from '@/components/source/queries';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import z from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';

export default function AddEditSourceDialog({
  sourcegroup,
  children,
  open,
  onOpenChange,
}: {
  sourcegroup?: SourceGroup;
  children?: React.ReactNode;
  open?: boolean;
  onOpenChange?: (open: boolean) => void;
}) {
  const [internalOpen, setInternalOpen] = useState(false);
  const { t } = useTranslation();
  const createSourceGroup = useCreateSourceGroup();
  const updateSourceGroup = useEditSourceGroup();
  const { data: sources } = useGetSources();

  // Use external state if provided, otherwise use internal state
  const isOpen = open !== undefined ? open : internalOpen;
  const handleOpenChange = onOpenChange || setInternalOpen;

  const sourceGroupSchema = z.object({
    id: z.number().optional(),
    name: z.string().min(1, t('Name is required')),
    description: z.string().max(255, t('Description must be 255 characters or less')).optional(),
    sources: z.array(z.any()),
  });

  type SourceGroupFormData = z.infer<typeof sourceGroupSchema>;

  const form = useForm<SourceGroupFormData>({
    resolver: zodResolver(sourceGroupSchema),
    values: sourcegroup
      ? {
          id: sourcegroup.id,
          name: sourcegroup.name,
          description: sourcegroup.description ?? '',
          sources: sourcegroup.sources || [],
        }
      : {
          id: undefined,
          name: '',
          description: '',
          sources: [],
        },
  });

  const selectedSources = form.watch('sources');

  const unselectedSources = useMemo(() => {
    return (sources || []).filter((src: Source) => !selectedSources.find((s: Source) => s.id === src.id));
  }, [sources, selectedSources]);

  const addSource = (source: Source) => {
    if (!selectedSources.find((s: Source) => s.id === source.id)) {
      form.setValue('sources', [...selectedSources, source]);
    }
  };

  const removeSource = (sourceId: number | undefined) => {
    form.setValue(
      'sources',
      selectedSources.filter((s: Source) => s.id !== sourceId)
    );
  };

  const onSubmit: SubmitHandler<SourceGroupFormData> = (data) => {
    const submitData = data as SourceGroup;

    if (sourcegroup) {
      updateSourceGroup.mutate(submitData, {
        onSuccess: () => {
          handleDialogChange(false);
        },
      });
    } else {
      createSourceGroup.mutate(submitData, {
        onSuccess: () => {
          handleDialogChange(false);
        },
      });
    }
  };

  const handleDialogChange = (isOpen: boolean) => {
    handleOpenChange(isOpen);
    if (!isOpen) {
      form.reset();
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleDialogChange}>
      {children && <DialogTrigger asChild>{children}</DialogTrigger>}
      <DialogContent className="p-0 sm:max-w-[800px]">
        <DialogHeader className="bg-card p-6">
          <DialogTitle>{sourcegroup ? t('Edit Flow Source Group') : t('Create Flow Source Group')}</DialogTitle>
          <DialogDescription>
            {t('You will need to add one or more source(s) to a sourcegroup. Fill out all information below.')}
          </DialogDescription>
        </DialogHeader>
        <ScrollArea className="h-[600px]">
          <Form {...form}>
            <form id="sourcegroup-form" onSubmit={form.handleSubmit(onSubmit)}>
              <div className="flex flex-col gap-2 p-6">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem className="flex w-full flex-wrap items-center gap-4">
                      <div className="space-y-1 md:w-2/3">
                        <FormLabel>{t('Flow Source Group Name')}</FormLabel>
                        <FormDescription>{t('Name of the flow source group, must be a unique name.')}</FormDescription>
                      </div>
                      <div className="relative w-full flex-1">
                        <FormControl className="flex-1">
                          <div className="col-span-2 w-full">
                            <Input {...field} />
                          </div>
                        </FormControl>
                        <FormMessage />
                      </div>
                    </FormItem>
                  )}
                />
              </div>
              <div className="flex flex-col gap-2 p-6">
                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem className="flex w-full flex-wrap items-center gap-4">
                      <div className="space-y-1 md:w-2/3">
                        <FormLabel>{t('Description')}</FormLabel>
                        <FormDescription>{t('Description of the flow source group.')}</FormDescription>
                      </div>
                      <div className="relative w-full flex-1">
                        <FormControl className="flex-1">
                          <div className="col-span-2 max-h-[200px] w-full">
                            <Textarea {...field} className="max-h-[200px]" maxLength={255} />
                          </div>
                        </FormControl>
                        <FormMessage />
                      </div>
                    </FormItem>
                  )}
                />
              </div>
              <div className="flex flex-col gap-2 p-6">
                <FormField
                  control={form.control}
                  name="sources"
                  render={() => (
                    <FormItem>
                      <div className="space-y-1">
                        <FormLabel>{t('Sources')}</FormLabel>
                        <FormDescription>{t('Select the sources you want in this flow source group.')}</FormDescription>
                      </div>
                      <div className="flex flex-col gap-4 md:flex-row">
                        {/* Left - Available Sources */}
                        <div className="bg-card w-full rounded-lg border p-4 md:w-1/2">
                          <h4 className="mb-2 font-semibold">{t('Available Sources')}</h4>
                          <ScrollArea className="h-60 overflow-x-hidden rounded-md">
                            <div className="space-y-2 p-1 pr-4">
                              {unselectedSources.map((source) => (
                                <div
                                  key={source.id}
                                  onClick={() => addSource(source)}
                                  className="bg-background hover:bg-muted cursor-pointer rounded p-2 shadow transition"
                                >
                                  <Tooltip>
                                    <TooltipTrigger asChild>
                                      <span
                                        className={`mr-2 inline-block h-2 w-2 rounded-full ${
                                          source.status ? 'bg-success' : 'bg-error'
                                        }`}
                                      />
                                    </TooltipTrigger>
                                    <TooltipContent>
                                      <p>{source.status ? t('Active') : t('Disabled')}</p>
                                    </TooltipContent>
                                  </Tooltip>
                                  {source.name}
                                </div>
                              ))}
                              {unselectedSources.length === 0 && (
                                <div className="text-muted-foreground text-sm">{t('No sources available.')}</div>
                              )}
                            </div>
                          </ScrollArea>
                        </div>

                        {/* Right - Selected Sources */}
                        <div className="bg-card w-full rounded-lg border p-4 md:w-1/2">
                          <h4 className="mb-2 font-semibold">{t('Selected Sources')}</h4>
                          <ScrollArea className="h-50 overflow-x-hidden rounded-md">
                            <div className="space-y-2 p-1 pr-4">
                              {selectedSources.map((source: Source) => (
                                <div
                                  key={source.id}
                                  onClick={() => removeSource(source.id)}
                                  className="bg-background hover:bg-muted cursor-pointer rounded p-2 shadow transition"
                                >
                                  <Tooltip>
                                    <TooltipTrigger asChild>
                                      <span
                                        className={`mr-2 inline-block h-2 w-2 rounded-full ${
                                          source.status ? 'bg-success' : 'bg-error'
                                        }`}
                                      />
                                    </TooltipTrigger>
                                    <TooltipContent>
                                      <p>{source.status ? t('Active') : t('Disabled')}</p>
                                    </TooltipContent>
                                  </Tooltip>
                                  {source.name}
                                </div>
                              ))}
                              {selectedSources.length === 0 && (
                                <div className="text-muted-foreground text-sm">{t('No sources selected.')}</div>
                              )}
                            </div>
                          </ScrollArea>
                        </div>
                      </div>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>
            </form>
          </Form>
        </ScrollArea>
        <DialogFooter className="bg-card p-6">
          <Button type="button" variant="secondary" onClick={() => handleDialogChange(false)}>
            {t('Cancel')}
          </Button>
          <Button type="submit" form="sourcegroup-form">
            {sourcegroup ? t('Update Flow Source Group') : t('Create Flow Source Group')}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
