import { ColumnDef } from '@tanstack/react-table';
import { Tooltip, TooltipContent, TooltipTrigger, TooltipProvider } from '@/components/ui/tooltip';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { useState, useRef, useEffect } from 'react';
import type { SuricataAlert } from '@/components/suricata/alerts/types';
import TextWithCopyButton from '@/components/ui/TextWithCopyButton';

function Cell({
  accessorKey,
  severity,
  useCopyButton,
  value,
}: {
  accessorKey: string;
  severity: string;
  useCopyButton: boolean;
  value: string;
}) {
  const textRef = useRef<HTMLDivElement>(null);
  const [isOverflowing, setIsOverflowing] = useState(false);

  useEffect(() => {
    const checkOverflow = () => {
      const el = textRef.current;
      if (el) {
        const isOver = el.scrollWidth > el.clientWidth;
        setIsOverflowing(isOver);
      }
    };

    let resizeTimeout: NodeJS.Timeout;
    const handleResize = () => {
      clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(() => {
        checkOverflow();
      }, 100);
    };

    checkOverflow();
    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
      clearTimeout(resizeTimeout);
    };
  }, [value]);

  const TextContent = (
    <div ref={textRef} className="w-full max-w-[300px] truncate overflow-hidden pl-1 text-ellipsis whitespace-nowrap">
      {value}
      {accessorKey === 'severity' && (
        <span
          className={`ml-2 inline-block h-3 w-3 rounded-full ${
            severity === '3' ? 'bg-success' : severity === '2' ? 'bg-warning' : 'bg-error'
          }`}
        />
      )}
    </div>
  );

  return isOverflowing ? (
    <TooltipProvider>
      <Tooltip>
        <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
        <TooltipContent>
          <p>{value}</p>
        </TooltipContent>
      </Tooltip>
    </TooltipProvider>
  ) : useCopyButton ? (
    <TextWithCopyButton text={value} />
  ) : (
    TextContent
  );
}

const createColumn = <T extends SuricataAlert>(
  accessorKey: keyof T,
  title: string,
  useCopyButton: boolean = false
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => <DataTableColumnHeader className="text-muted-foreground" column={column} title={title} />,
  cell: ({ row }) => (
    <Cell
      accessorKey={accessorKey as string}
      severity={row.original.severity}
      useCopyButton={useCopyButton}
      value={row.getValue(accessorKey as string)}
    />
  ),
});
// In the future being able to disable a rule from these columns could be nice

export const alertColumns: ColumnDef<SuricataAlert>[] = [
  createColumn('SID', 'SID', true),
  createColumn('count', 'Count'),
  createColumn('signature', 'Signature'),
  createColumn('category', 'Category'),
  createColumn('severity', 'Severity'),
  createColumn('Last Seen', 'Last Seen'),
];
