import { useTranslation } from "react-i18next";
import { useGetConfig, useUpdateConfig } from "@/components/suricata/config/queries";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  Form,
  FormDescription,
} from "@/components/ui/form";
import { useForm, SubmitHandler } from "react-hook-form";
import { SuricataConfigData } from "@/components/suricata/config/types";
import { 
  Card,
  CardContent,
} from "@/components/ui/card"
import { PageSeparator } from "@/components/Page";

export function SuricataConfig() {
  const { t } = useTranslation();
  const { data: currentConfig, isLoading } = useGetConfig();
  const updateConfig = useUpdateConfig();
  
  const onSubmit: SubmitHandler<SuricataConfigData> = async (data) => {
    await updateConfig.mutateAsync({data, update: false});
  };

  const form = useForm<SuricataConfigData>({
    defaultValues: {
      suricata_log_path: currentConfig
        ? currentConfig.suricata_log_path
        : "",
      suricata_rules_path: currentConfig
        ? currentConfig.suricata_rules_path
        : "",
      suricata_yaml_path: currentConfig
        ? currentConfig.suricata_yaml_path
        : "",
    },
    values: currentConfig,
  });

  const { isDirty } = form.formState;

  if (isLoading) {
    return <div>{t("Loading configuration...")}</div>;
  }

  return (
    <Card>
      <CardContent>
      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(onSubmit)}
        >
          <div className="flex flex-row items-center justify-between">
            <h1 className="leading-9 text-md font-large font-semibold">
              {t("Suricata Configuration")}
            </h1>
            <Button
              disabled={!isDirty}
              variant={isDirty ? "default" : "outline"}
              className="border"
              type="submit"
            >
              {t("Update")}
            </Button>
          </div>
          <PageSeparator/>
          <FormField
            control={form.control}
            name="suricata_log_path"
            render={({ field }) => (
              <FormItem className="flex items-center justify-between flex-wrap gap-4">
                  <div className="space-y-1">
                    <FormLabel>{t("Suricata Log Path Directory")}</FormLabel>
                    <FormDescription>
                      {t("Path to the directory where Suricata logs are stored.")}
                    </FormDescription>
                  </div>
                <FormControl className="w-md">
                  <Input {...field} placeholder="/path/to/logs/"/>
                </FormControl>
              </FormItem>
            )}
          />
          <PageSeparator />
          <FormField
            control={form.control}
            name="suricata_rules_path"
            render={({ field }) => (
              <FormItem className="flex items-center justify-between flex-wrap gap-4">
                  <div className="space-y-1">
                    <FormLabel>{t("Suricata Rules Path")}</FormLabel>
                    <FormDescription>
                      {t("Path to the directory where Suricata rules are stored.")}
                    </FormDescription>
                  </div>
                <FormControl className="w-md">
                  <Input {...field} placeholder="/path/to/rules/"/>
                </FormControl>
              </FormItem>
            )}
          />
          <PageSeparator />
          <FormField
            control={form.control}
            name="suricata_yaml_path"
            render={({ field }) => (
              <FormItem className="flex items-center justify-between flex-wrap gap-4">
                  <div className="space-y-1">
                    <FormLabel>{t("Suricata Yaml Path")}</FormLabel>
                    <FormDescription>
                      {t("Path to the directory where the Suricata yaml file is stored.")}
                    </FormDescription>
                  </div>
                <FormControl className="w-md">
                  <Input {...field} placeholder="/path/to/yaml/"/>
                </FormControl>
              </FormItem>
            )}
          />
        </form>
      </Form>
      </CardContent>
    </Card>
  );
}
