import axios from "@/lib/axios";
import { useQuery, useQueryClient, useMutation } from "@tanstack/react-query";
import { toast } from "sonner";
import {
  SuricataConfigData,
} from "@/components/suricata/config/types";
import { useTranslation } from "react-i18next";


export function useGetConfig() {
  return useQuery<SuricataConfigData>({
    queryKey: ["suricataConfig"],
    queryFn: async () => {
      const response = await axios.get("/api/v1/suricata/config");
      return response.data;
    },
  });
}
 
export function useUpdateConfig() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ data }: { data: SuricataConfigData; update: boolean }) =>
      axios.put("/api/v1/suricata/config", {
        evelog_name: data.evelog_name,
        suricata_log_path: data.suricata_log_path,
        suricata_rules_name: data.suricata_rules_name,
        suricata_rules_path: data.suricata_rules_path,
        suricata_yaml_path: data.suricata_yaml_path,
      }),
    onSuccess: (_, context) => {
      queryClient.invalidateQueries({ queryKey: ["suricataConfig"] });
      queryClient.invalidateQueries({ queryKey: ["AllSuricataData"] });
      queryClient.invalidateQueries({ queryKey: ["JsonFiles"] });
      queryClient.invalidateQueries({ queryKey: ["suricataRules"] });
      queryClient.invalidateQueries({queryKey: ["RulesFiles"]});
      const { update } = context;
      if (!update)
        toast.success(t("Configuration Saved Successfully."));
    },
    onError: () => {
      toast.error(t("Configuration Failed to Save."));
    },
    retry: 3,
    retryDelay: attemptIndex => Math.min(1000 * attemptIndex, 5000), 
  });
}

export function useUpdateAutoRuleset() {
  return useMutation({
    mutationFn: (data: boolean) => 
      axios.put("/api/v1/suricata/config", {
          auto_update_rules: data
      })
  })
}

export function useUpdateDataRetentionTime() {
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (data: string) => 
      axios.put("/api/v1/suricata/config", {
          data_retention_time: data
      }),
    onSuccess: () => {
      toast.success(t("Configuration Saved Successfully."));
    },
    onError: () => {
      toast.error(t("Configuration Failed to Save."));
    }
  })
}

export function useUpdateRulesetInterval() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (data: string) =>
      axios.put("/api/v1/suricata/config", {
          ruleset_update_interval: data
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["suricataConfig"] });
      toast.success(t("Configuration Saved Successfully."));
    },
    onError: () => {
      toast.error(t("Configuration Failed to Save."))
    }
  })
}
