import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogTrigger,
  DialogClose,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useTranslation } from 'react-i18next';
import { Settings } from 'lucide-react';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { Form, FormControl, FormItem, FormLabel, FormMessage, FormDescription, FormField } from '@/components/ui/form';
import { useUpdateDataRetentionTime, useGetConfig } from '@/components/suricata/config/queries';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';

export function ConfigDialog() {
  const { t } = useTranslation();
  const [open, setOpen] = useState(false);
  const { mutate: updateDataRetentionTime } = useUpdateDataRetentionTime();
  const { data: configData } = useGetConfig();
  const timeOptions = [
    { value: '1h', label: t('1 Hour') },
    { value: '6h', label: t('6 Hour') },
    { value: '12h', label: t('12 Hour') },
    { value: '24h', label: t('1 Day') },
    { value: '48h', label: t('2 Days') },
    { value: '1w', label: t('1 Week') },
    { value: '1m', label: t('1 Month') },
    { value: 'never', label: t('Never') },
  ];

  const form = useForm({
    values: {
      data_retention_time: configData?.data_retention_time || '24h',
    },
  });

  const onSubmit = form.handleSubmit((values) => {
    updateDataRetentionTime(values.data_retention_time);
    setOpen(false);
  });

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <Form {...form}>
        <form>
          <Tooltip>
            <TooltipTrigger asChild>
              <DialogTrigger asChild>
                <Button size="icon" variant="outline">
                  <Settings />
                </Button>
              </DialogTrigger>
            </TooltipTrigger>
            <TooltipContent>{t('Suricata Data Configuration')}</TooltipContent>
          </Tooltip>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>{t('Suricata Data Configuration')}</DialogTitle>
              <DialogDescription>{t('Configure your Suricata data here')}</DialogDescription>
            </DialogHeader>
            <FormField
              control={form.control}
              name="data_retention_time"
              render={({ field }) => (
                <FormItem className="mb-5">
                  <FormLabel>{t('Data Retention Time')}</FormLabel>
                  <FormControl>
                    <Select value={field.value} onValueChange={field.onChange}>
                      <SelectTrigger className="w-full">
                        <SelectValue placeholder={t('Select time period')} />
                      </SelectTrigger>
                      <SelectContent>
                        {timeOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </FormControl>
                  <FormDescription>
                    {t(`Select the time period data will be retained from the main suricata process in the database.`)}
                    <br />
                    <span className="font-bold">
                      {t(
                        `NOTE: This will not affect the eve log file or any scanned PCAP files stored in the database.`
                      )}
                    </span>
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />
            <DialogFooter>
              <DialogClose asChild>
                <Button variant="secondary">{t('Close')}</Button>
              </DialogClose>
              <Button onClick={onSubmit}>{t('Save')}</Button>
            </DialogFooter>
          </DialogContent>
        </form>
      </Form>
    </Dialog>
  );
}
